<?php

namespace App\Controladores;

use App\Modelos\Empleado;
use Core\Controller;
use Core\Librerias\CarritoVentas;
use Core\Librerias\Http;
use Core\Librerias\Seguridad;
use Core\Vista;

final class FacturacioncajaControlador extends Controller
{
    private $array_errors = [];
    private $carritoVentas;

    public function __construct()
    {
        parent::__construct();
        $this->carritoVentas = new CarritoVentas();
    }

    public function nuevaFactura(int $id_cliente)
    {
        if ('GET' == $_SERVER['REQUEST_METHOD']) {

            $id_cliente = filter_var($id_cliente, FILTER_SANITIZE_NUMBER_INT);
            $cliente = $this->carritoVentas->obtenerCliente($id_cliente);
            $_SESSION["factura$cliente[cli_id]"] = null;

            if ($cliente) {
                return Vista::view('facturacioncaja/index', [
                    'cliente'               => $cliente,
                    'doc_pendientes'        => $this->carritoVentas->totalDocumentosPendientes($id_cliente),
                    'empresa'               => $this->carritoVentas->obtenerDatosEmpresa(),
                    'tipos_de_documentos'   => $this->carritoVentas->obternerTiposDeDocumentos(),
                    'impuestos'             => $this->carritoVentas->obtenerImpuestos(),
                    'paises'                => $this->carritoVentas->obtenerPaises(),
                    'almacenes'             => $this->carritoVentas->obtenerAlmacenesEmpleado(),
                    'series'                => $this->carritoVentas->obtenerSeriesEmpleado(),
                    'divisas'               => $this->carritoVentas->obtenerDivisasParaFacturar(),
                    'empleados'             => $this->carritoVentas->obtenerEmpleados()
                ]);
            } else {
                header('Location: ' . ruta_base() . 'facturacion');
            }
        }
    }

    public function buscarProducto()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_GET['divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';
            if (empty($_GET['almacen']))
                $this->array_errors[] = 'El almacen es obligatorio';

            if (empty($this->array_errors)) {

                $id_divisa      = filter_input(INPUT_GET, 'divisa', FILTER_VALIDATE_INT);
                $id_almacen     = filter_input(INPUT_GET, 'almacen', FILTER_VALIDATE_INT);
                $producto       = $this->solicitud->sanitize($_GET['buscar'], FILTER_SANITIZE_STRING);

                $result = $this->carritoVentas->buscarProducto($producto, $id_divisa, $id_almacen, true, true);
                if (isset($result['errors'])) {
                    return Http::json_response(['errors' => $result['errors']]);
                } else {
                    return Http::json_response($result);
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function predeterminarDivisa()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_GET['id_cliente']))
                $this->array_errors[] = 'El Cliente es obligatorio';
            if (empty($_GET['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_divisa          = filter_input(INPUT_GET, 'id_divisa', FILTER_VALIDATE_INT);
                $id_cliente         = filter_input(INPUT_GET, 'id_cliente', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_GET, 'descuento', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_cliente && $id_divisa) {
                    $result = $this->carritoVentas->predeterminarDivisa('factura', $id_cliente, $id_divisa, $descuento_global);
                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function agregarProductoBarcode()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_cliente']))
                $this->array_errors[] = 'El Cliente es obligatorio';
            if (empty($_POST['producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['id_almacen']))
                $this->array_errors[] = 'El Almacen es obligatorio';
            if (empty($_POST['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_cliente         = filter_input(INPUT_POST, 'id_cliente', FILTER_VALIDATE_INT);
                $producto           = filter_input(INPUT_POST, 'producto', FILTER_SANITIZE_STRIPPED);
                $id_almacen         = filter_input(INPUT_POST, 'id_almacen', FILTER_VALIDATE_INT);
                $id_divisa          = filter_input(INPUT_POST, 'id_divisa', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_cliente && $id_almacen && $id_divisa) {
                    $result = $this->carritoVentas->agregarProductoBarcode('app_factura_master', 'factura', $id_cliente, $producto, $id_almacen, $id_divisa, $descuento_global, true, true);
                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function agregarProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_cliente']))
                $this->array_errors[] = 'El Cliente es obligatorio';
            if (empty($_POST['id_producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['id_almacen']))
                $this->array_errors[] = 'El Almacen es obligatorio';
            if (empty($_POST['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_cliente         = filter_input(INPUT_POST, 'id_cliente', FILTER_VALIDATE_INT);
                $id_producto        = filter_input(INPUT_POST, 'id_producto', FILTER_VALIDATE_INT);
                $id_almacen         = filter_input(INPUT_POST, 'id_almacen', FILTER_VALIDATE_INT);
                $id_divisa          = filter_input(INPUT_POST, 'id_divisa', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_cliente && $id_producto && $id_almacen && $id_divisa) {
                    $result = $this->carritoVentas->agregarProducto('app_factura_master', 'factura', $id_cliente, $id_producto, $id_almacen, $id_divisa, $descuento_global);
                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarCantidad()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_cliente']))
                $this->array_errors[] = 'El Cliente es obligatorio';
            if (empty($_POST['id_producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['cantidad']))
                $this->array_errors[] = 'El Cantidad es obligatorio';

            if (empty($this->array_errors)) {

                $id_cliente         = filter_input(INPUT_POST, 'id_cliente', FILTER_VALIDATE_INT);
                $id_producto        = filter_input(INPUT_POST, 'id_producto', FILTER_VALIDATE_INT);
                $cantidad           = filter_input(INPUT_POST, 'cantidad', FILTER_VALIDATE_FLOAT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_cliente && $id_producto && $cantidad) {
                    $result = $this->carritoVentas->actualizarCantidad('factura', $id_cliente, $id_producto, $cantidad, $descuento_global);
                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function aplicarDescuentoAdicional()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_cliente'])) {
                $this->array_errors[] = 'El Cliente es obligatorio';
            } else {

                $id_cliente = filter_input(INPUT_POST, 'id_cliente', FILTER_VALIDATE_INT);
                $descuento_global = filter_input(INPUT_POST, 'descuento', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_cliente) {
                    $result = $this->carritoVentas->aplicarDescuentoAdicional('factura', $id_cliente, $descuento_global);
                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function eliminarProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_cliente']))
                $this->array_errors[] = 'El Cliente es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';

            if (empty($this->array_errors)) {

                $id_cliente         = filter_input(INPUT_POST, 'id_cliente', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRING);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_cliente) {
                    $result = $this->carritoVentas->eliminarProducto('factura', $id_cliente, $unique, $descuento_global);
                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function guardarDetallesCliente()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['cli_tipodoc']))
                $this->array_errors[] = 'El campo Cedula de identidad es obligatorio';
            if (empty($_POST['cli_dni']))
                $this->array_errors[] = 'El campo Tipo de documento es obligatorio';
            if (empty($_POST['cli_razon_social']))
                $this->array_errors[] = 'El campo Razón social es obligatorio';
            if (empty($_POST['cli_direccion']))
                $this->array_errors[] = 'El campo Dirección es obligatorio';

            if (empty($this->array_errors)) {

                $cli_id             = filter_input(INPUT_POST, 'id', FILTER_SANITIZE_NUMBER_INT);
                $cli_tipodoc        = filter_input(INPUT_POST, 'cli_tipodoc', FILTER_SANITIZE_NUMBER_INT);
                $cli_dni            = filter_input(INPUT_POST, 'cli_dni', FILTER_SANITIZE_STRING);
                $cli_nombres        = $this->solicitud->sanitize($_POST['cli_razon_social'], FILTER_SANITIZE_STRING);
                $cli_direccion      = $this->solicitud->sanitize($_POST['cli_direccion'], FILTER_SANITIZE_STRING);
                $cli_pais           = filter_input(INPUT_POST, 'cli_paisid', FILTER_SANITIZE_NUMBER_INT);
                $cli_estado         = $this->solicitud->sanitize($_POST['cli_estado'], FILTER_SANITIZE_STRING);
                $cli_ciudad         = $this->solicitud->sanitize($_POST['cli_ciudad'], FILTER_SANITIZE_STRING);
                $cli_codigopostal   = filter_input(INPUT_POST, 'cli_codigopostal', FILTER_SANITIZE_STRING);

                $result = $this->carritoVentas->editarDetallesCliente($cli_id, [
                    'cli_tipodocid'     => $cli_tipodoc,
                    'cli_dni'           => $cli_dni,
                    'cli_razon_social'  => $cli_nombres,
                    'cli_direccion'     => $cli_direccion,
                    'cli_paisid'        => $cli_pais,
                    'cli_estado'        => $cli_estado,
                    'cli_ciudad'        => $cli_ciudad,
                    'cli_codpostal'     => $cli_codigopostal
                ]);

                if (isset($result['errors'])) {
                    return Http::json_response(['errors' => $result['errors']]);
                } else {
                    return Http::json_response($result);
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function cambiarAlmacen(int $id_cliente)
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $_SESSION["factura$id_cliente"] = null;
            Http::json_response(true);
        } else {
            Http::json_response(false);
        }
    }

    public function ventaEnEspera()
    {
        if ($_SERVER['REQUEST_METHOD'] == 'POST' && isAjax()) {

            if (empty($_POST['id_serie']))
                $this->array_errors[] = 'El campo Serie es obligatorio.';
            if (empty($_POST['fecha']))
                $this->array_errors[] = 'Debe seleccionar una fecha.';
            if (empty($_POST['id_cliente']))
                $this->array_errors[] = 'Acción prohibida';

            if (empty($this->array_errors)) {

                $id_cliente         = filter_input(INPUT_POST, 'id_cliente', FILTER_VALIDATE_INT);
                $id_serie           = filter_input(INPUT_POST, 'id_serie', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;
                $fecha              = filter_input(INPUT_POST, 'fecha', FILTER_SANITIZE_STRING);
                $observaciones      = filter_input(INPUT_POST, 'observaciones', FILTER_SANITIZE_STRING) ?? '';

                if (!$id_cliente || !$id_serie)
                    $this->array_errors[] = 'Datos incorrectos';

                if (empty($this->array_errors)) {

                    $result = $this->carritoVentas->guardarVenta('app_factura_master', 'factura', $id_cliente, $id_serie, $fecha, $observaciones, true, 0, $descuento_global, true);

                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function autorizarDescuentoProducto()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['empleado']))
                $this->array_errors[] = 'El empleado es obligatorio';
            if (empty($_POST['clave']))
                $this->array_errors[] = 'La clave es obligatorio';
            if (empty($_POST['id_cliente']))
                $this->array_errors[] = 'Acción prohibida';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'Acción prohibida';

            if (empty($this->array_errors)) {

                $id_cliente = filter_var($_POST['id_cliente'], FILTER_VALIDATE_INT);
                $descuento = filter_var($_POST['descuento'], FILTER_SANITIZE_STRING) ?? 0;
                $descuento_global = filter_var($_POST['descuento_global'], FILTER_SANITIZE_STRING) ?? 0;
                $empleado = $this->solicitud->sanitize($_POST['empleado'], FILTER_SANITIZE_STRING);
                $clave = $this->solicitud->sanitize($_POST['clave'], FILTER_SANITIZE_STRING);
                $unique = $_POST['unique'];

                if (!$id_cliente)
                    $this->array_errors[] = 'Acción prohibida';
                if (!$descuento)
                    $this->array_errors[] = 'El descuento debe ser de tipo numerico';

                if (empty($this->array_errors)) {

                    $cliente = $this->carritoVentas->obtenerCliente($id_cliente);

                    if ($cliente) {

                        $empleado_model = new Empleado();
                        $usuario = $empleado_model->getUserInfo($empleado);

                        if (!isset($usuario['emp_id'])) {
                            $this->array_errors[] = 'Usuario no encontrado';
                        } else {
                            if (!Seguridad::verificar($clave, $usuario['emp_password'])) {
                                $this->array_errors[] = 'Clave incorrecta';
                            }
                        }

                        if (empty($this->array_errors)) {
                            $result = $this->carritoVentas->aplicarDescuentoProducto('factura', $id_cliente, $unique, $descuento, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }
}
