<?php

namespace App\Controladores;

use App\Modelos\Almacenes;
use App\Modelos\CarritoVentasModel;
use App\Modelos\Cmpfactura;
use App\Modelos\Cmpnotas;
use App\Modelos\CmpPedidos;
use App\Modelos\Compras;
use App\Modelos\Ingresosegresos;
use App\Modelos\Productos;
use App\Modelos\ProductosCompuestos;
use App\Modelos\Proveedor;
use Core\Controller;
use Core\Helpers\Moneda;
use Core\Librerias\CarritoCompras;
use Core\Librerias\Http;
use Core\Librerias\Module;
use Core\Vista;
use DateTime;
use DateTimeZone;
use IntlChar;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Reader\Xls\Style\Border;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Border as StyleBorder;
use PhpOffice\PhpSpreadsheet\Style\Color;
use Rakit\Validation\Rules\Integer;
use Rakit\Validation\Rules\Numeric;

final class ComprasControlador extends Controller
{
    private $compras;
    private $array_errors = [];
    private $module;
    private $div_master;
    private $ingresosegresos_model;

    public function __construct()
    {
        parent::__construct();
        $this->module   = new Module('compras');
        $this->compras  = new Compras();
        $this->ingresosegresos_model  = new Ingresosegresos();
        $this->div_master               = $this->compras->obtenerDivisaMaster();
    }

    public function index()
    {
        $newProv = new Proveedor();
        $newAlmacen = new Almacenes();

        return Vista::view('compras/index', [
            'tipos_de_documentos' => $this->compras->obtenerTiposdeDocumentos(),
            'divisas'               => $this->ingresosegresos_model->obtenerDivisas(),
            'cmppp'                 => $this->obtenerCMPPP(),
            'proveedores' => $newProv->proveedoresActivos(),
            'almacenes' => $newAlmacen->almacenesActivas(),
            'cmpnotascreditos' => $this->module->has_module_action_permission('compras', 'cmpnotascreditos', $_SESSION['user_data']['emp_id']),
            'cmpnotasentrega' => $this->module->has_module_action_permission('compras', 'cmpnotasentrega', $_SESSION['user_data']['emp_id']),
            'cmpfacturas' => $this->module->has_module_action_permission('compras', 'cmpfacturas', $_SESSION['user_data']['emp_id']),
            'cmppedido' => $this->module->has_module_action_permission('compras', 'cmppedido', $_SESSION['user_data']['emp_id']),
            'reporte_general' => $this->module->has_module_action_permission('compras', 'reporte_general', $_SESSION['user_data']['emp_id']),
            'crear_pedido' => $this->module->has_module_action_permission('compras', 'crear_pedido', $_SESSION['user_data']['emp_id']),
            'sucursales' => $this->compras->consultar("SELECT * FROM app_empleados_sucursales s JOIN app_sucursales su ON s.sucursal_id = su.su_id WHERE empleado_id={$_SESSION['user_data']['emp_id']}")->all(),
            'facturas_compras' => $this->module->has_module_action_permission('ingresosegresos', 'facturas_compras', $_SESSION['user_data']['emp_id']),
            'cuentas_pagar' => $this->module->has_module_action_permission('ingresosegresos', 'cuentas_pagar', $_SESSION['user_data']['emp_id']),
            'proveedores_frecuente' => $this->module->has_module_action_permission('compras', 'proveedores_frecuente', $_SESSION['user_data']['emp_id']),
        ]);
    }

    public function obtenerDivisaPredeterminada()
    {
        Http::json_response(Moneda::Predeterminada());
    }

    public function buscarProveedores()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $buscar = filter_input(INPUT_GET, 'buscar', FILTER_SANITIZE_STRING);
            Http::json_response($this->compras->buscarProveedores($buscar));
        }
    }

    public function obtenerFacturas()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $anio = filter_input(INPUT_GET, 'anio', FILTER_SANITIZE_NUMBER_INT);
            $mes = $this->solicitud->sanitize($_GET['mes'], FILTER_SANITIZE_NUMBER_INT);

            $array_anuladas     = [];
            $array_pendientes   = [];
            $array_pagas        = [];
            if ($mes == 0) {
                $f_pagas = $this->compras->obtenerFacturasAnuales("YEAR(cmp.cmp_fecha)=$anio");
                $f_pendientes = $this->compras->obtenerFacturasAnualesPendientes("YEAR(cmp.cmp_fecha)=$anio");
                $f_anuladas = $this->compras->obtenerFacturasAnualesAnuladas("YEAR(cmp.cmp_fecha)=$anio");
            } else {
                $f_pagas = $this->compras->obtenerFacturasAnuales("MONTH(cmp.cmp_fecha)=$mes AND YEAR(cmp.cmp_fecha)=$anio");
                $f_pendientes = $this->compras->obtenerFacturasAnualesPendientes("MONTH(cmp.cmp_fecha)=$mes AND YEAR(cmp.cmp_fecha)=$anio");
                $f_anuladas = $this->compras->obtenerFacturasAnualesAnuladas("MONTH(cmp.cmp_fecha)=$mes AND YEAR(cmp.cmp_fecha)=$anio");
            }


            if (is_countable($f_pagas) && count($f_pagas)) {
                foreach ($f_pagas as $row) {
                    $array_pagas[] = [
                        'id'            => $row['cmp_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmp_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmp_total'],
                        'fecha'         => $row['cmp_fecha'],
                        'factura'       => $row['cmp_numfactura']
                    ];
                }
            }

            if (is_countable($f_pendientes) && count($f_pendientes)) {
                foreach ($f_pendientes as $row) {
                    $acumulado = $this->compras->obtenerAcumulado($row['cmp_id'], 8);
                    $acumulado_total = ($acumulado['total']) ?? 0;
                    $array_pendientes[] = [
                        'id'            => $row['cmp_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmp_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmp_total'],
                        'fecha'         => $row['cmp_fecha'],
                        'acumulado'     => $acumulado_total * -1,
                        'pendiente'     => $row['cmp_total'] + $acumulado_total,
                        'factura'       => $row['cmp_numfactura']
                    ];
                }
            }

            if (is_countable($f_anuladas) && count($f_anuladas)) {
                foreach ($f_anuladas as $row) {
                    $acumulado = $this->compras->obtenerAcumulado($row['cmp_id'], 8);
                    $acumulado_total = ($acumulado['total']) ?? 0;
                    $array_anuladas[] = [
                        'id'            => $row['cmp_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmp_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmp_total'],
                        'fecha'         => $row['cmp_fecha'],
                        'acumulado'     => $acumulado_total * -1,
                        'pendiente'     => $row['cmp_total'] + $acumulado_total,
                        'factura'       => $row['cmp_numfactura']
                    ];
                }
            }

            Http::json_response([
                'pagas'         => $array_pagas,
                'pendientes'    => $array_pendientes,
                'anuladas'      => $array_anuladas
            ]);
        }
    }

    public function obtenerNotasdeEntregaAnuales()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $anio = filter_input(INPUT_GET, 'anio', FILTER_SANITIZE_NUMBER_INT);
            $mes = $this->solicitud->sanitize($_GET['mes'], FILTER_SANITIZE_NUMBER_INT);

            $array_pendientes   = [];
            $array_anuladas     = [];
            $array_anuales      = [];

            if ($mes == 0) {
                $n_pendientes = $this->compras->obtenerNotasdeEntregaAnualesPendientes("YEAR(cmp.cmpn_fecha)=$anio");
                $n_anuladas = $this->compras->obtenerNotasdeEntregaAnualesAnuladas("YEAR(cmp.cmpn_fecha)=$anio");
                $n_anuales = $this->compras->obtenerNotasdeEntregaAnuales("YEAR(cmp.cmpn_fecha)=$anio");
            } else {
                $n_pendientes = $this->compras->obtenerNotasdeEntregaAnualesPendientes("MONTH(cmp.cmpn_fecha)=$mes AND YEAR(cmp.cmpn_fecha)=$anio");
                $n_anuladas = $this->compras->obtenerNotasdeEntregaAnualesAnuladas("MONTH(cmp.cmpn_fecha)=$mes AND YEAR(cmp.cmpn_fecha)=$anio");
                $n_anuales = $this->compras->obtenerNotasdeEntregaAnuales("MONTH(cmp.cmpn_fecha)=$mes AND YEAR(cmp.cmpn_fecha)=$anio");
            }

            if (is_countable($n_pendientes) && count($n_pendientes)) {
                foreach ($n_pendientes as $row) {
                    $acumulado = $this->compras->obtenerAcumulado($row['cmpn_id'], 9);
                    $acumulado_total = ($acumulado['total']) ?? 0;
                    $array_pendientes[] = [
                        'id'            => $row['cmpn_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmpn_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmpn_total'],
                        'fecha'         => $row['cmpn_fecha'],
                        'acumulado'     => $acumulado_total * -1,
                        'pendiente'     => $row['cmpn_total'] + $acumulado_total,
                        'factura'       => $row['cmpn_numfactura']
                    ];
                }
            }


            if (is_countable($n_anuladas) && count($n_anuladas)) {
                foreach ($n_anuladas as $row) {
                    $acumulado = $this->compras->obtenerAcumulado($row['cmpn_id'], 9);
                    $acumulado_total = ($acumulado['total']) ?? 0;
                    $array_anuladas[] = [
                        'id'            => $row['cmpn_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmpn_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmpn_total'],
                        'fecha'         => $row['cmpn_fecha'],
                        'acumulado'     => $acumulado_total * -1,
                        'pendiente'     => $row['cmpn_total'] + $acumulado_total,
                        'factura'       => $row['cmpn_numfactura']
                    ];
                }
            }


            if (is_countable($n_anuales) && count($n_anuales)) {
                foreach ($n_anuales as $row) {
                    $array_anuales[] = [
                        'id'            => $row['cmpn_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmpn_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmpn_total'],
                        'fecha'         => $row['cmpn_fecha'],
                        'factura'       => $row['cmpn_numfactura']
                    ];
                }
            }

            Http::json_response([
                'pagas'         => $array_anuales,
                'pendientes'    => $array_pendientes,
                'anuladas'      => $array_anuladas,
            ]);
        }
    }

    public function obtenerPedidosAnuales()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $anio = filter_input(INPUT_GET, 'anio', FILTER_SANITIZE_NUMBER_INT);
            $mes = $this->solicitud->sanitize($_GET['mes'], FILTER_SANITIZE_NUMBER_INT);

            $array_pendientes   = [];
            $array_anuladas     = [];
            $array_anuales      = [];

            if ($mes == 0) {
                $n_pendientes = $this->compras->obtenerPedidosAnualesPendientes("YEAR(cmp.cmppe_fecha)=$anio");
                $n_anuladas = $this->compras->obtenerPedidosAnualesAnuladas("YEAR(cmp.cmppe_fecha)=$anio");
                $n_anuales = $this->compras->obtenerPedidosAnuales("YEAR(cmp.cmppe_fecha)=$anio");
            } else {
                $n_pendientes = $this->compras->obtenerPedidosAnualesPendientes("MONTH(cmp.cmppe_fecha)=$mes AND YEAR(cmp.cmppe_fecha)=$anio");
                $n_anuladas = $this->compras->obtenerPedidosAnualesAnuladas("MONTH(cmp.cmppe_fecha)=$mes AND YEAR(cmp.cmppe_fecha)=$anio");
                $n_anuales = $this->compras->obtenerPedidosAnuales("MONTH(cmp.cmppe_fecha)=$mes AND YEAR(cmp.cmppe_fecha)=$anio");
            }

            if (is_countable($n_pendientes) && count($n_pendientes)) {
                foreach ($n_pendientes as $row) {
                    $acumulado = $this->compras->obtenerAcumulado($row['cmppe_id'], 10);
                    $acumulado_total = ($acumulado['total']) ?? 0;
                    $array_pendientes[] = [
                        'id'            => $row['cmppe_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmppe_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmppe_total'],
                        'fecha'         => $row['cmppe_fecha'],
                        'acumulado'     => $acumulado_total * -1,
                        'pendiente'     => $row['cmppe_total'] + $acumulado_total,
                        'factura'       => ''
                    ];
                }
            }


            if (is_countable($n_anuladas) && count($n_anuladas)) {
                foreach ($n_anuladas as $row) {
                    $acumulado = $this->compras->obtenerAcumulado($row['cmppe_id'], 10);
                    $acumulado_total = ($acumulado['total']) ?? 0;
                    $array_anuladas[] = [
                        'id'            => $row['cmppe_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmppe_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmppe_total'],
                        'fecha'         => $row['cmppe_fecha'],
                        'acumulado'     => $acumulado_total * -1,
                        'pendiente'     => $row['cmppe_total'] + $acumulado_total,
                        'factura'       => ''
                    ];
                }
            }


            if (is_countable($n_anuales) && count($n_anuales)) {
                foreach ($n_anuales as $row) {
                    $array_anuales[] = [
                        'id'            => $row['cmppe_id'],
                        'cliente'       => $row['prov_razonsocial'],
                        'correlativo'   => $row['cmppe_correlativo'],
                        'div_locale'    => $row['div_locale'],
                        'div_simbolo'   => $row['div_simbolo'],
                        'total'         => $row['cmppe_total'],
                        'fecha'         => $row['cmppe_fecha'],
                        'factura'       => ''
                    ];
                }
            }

            Http::json_response([
                'pagas'         => $array_anuales,
                'pendientes'    => $array_pendientes,
                'anuladas'      => $array_anuladas,
            ]);
        }
    }

    public function buscarLineas()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $buscar = filter_input(INPUT_GET, 'buscar', FILTER_SANITIZE_STRING);
            $filtro = filter_input(INPUT_GET, 'filtro', FILTER_SANITIZE_NUMBER_INT);
            $buscar = strtolower($buscar);

            $timeZone = new DateTimeZone('America/Caracas');
            $fecha_desde = (new DateTime(null, $timeZone))->modify('first day of last month')->format('Y-m-d');
            $fecha_hasta = date('Y-m-d');

            $array_lineas = [];
            $notas = $this->compras->buscarLineasNEV(trim($buscar), $filtro, $fecha_desde, $fecha_hasta);

            foreach ($notas as $row) {

                $divisa = $this->compras->obtenerDivisa($row['cmpn_divid']);

                $array_lineas[] = [
                    'ruta_documento'    => ruta_base() . 'cmpnotaseditar/verFactura/' . $row['cmpn_id'],
                    'correlativo'       => 'NEC' . $row['cmpn_correlativo'],
                    'proveedor'         => $row['prov_razonsocial'],
                    'ruta_producto'     => ruta_base() . 'productos/editar/' . $row['cmpnart_proid'],
                    'id_pro'            => $row['cmpnart_proid'],
                    'cantidad'          => $row['cmpnart_procantidad'],
                    'art_codigo'        => $row['cmpnart_procodigo'],
                    'art_nombre'        => $row['cmpnart_propnombre'],
                    'total'             => Moneda::moneda($row['cmpnart_prototal'], $divisa['div_locale'], $divisa['div_simbolo']),
                    'fecha'             => $row['fecha']
                ];
            }

            $facuras = $this->compras->buscarLineasFACV(trim($buscar), $filtro, $fecha_desde, $fecha_hasta);

            foreach ($facuras as $row) {

                $divisa = $this->compras->obtenerDivisa($row['cmp_divid']);

                $array_lineas[] = [
                    'ruta_documento'    => ruta_base() . 'cmpfacturaeditar/verFactura/' . $row['cmp_id'],
                    'correlativo'       => 'FACC' . $row['cmp_correlativo'],
                    'proveedor'         => $row['prov_razonsocial'],
                    'ruta_producto'     => ruta_base() . 'productos/editar/' . $row['cmpart_proid'],
                    'id_pro'            => $row['cmpart_proid'],
                    'cantidad'          => $row['cmpart_procantidad'],
                    'art_codigo'        => $row['cmpart_procodigo'],
                    'art_nombre'        => $row['cmpart_propnombre'],
                    'total'             => Moneda::moneda($row['cmpart_prototal'], $divisa['div_locale'], $divisa['div_simbolo']),
                    'fecha'             => $row['fecha']
                ];
            }

            Http::json_response($array_lineas);
        }
    }

    public function guardarProveedor()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $ruta                   = filter_input(INPUT_POST, 'ruta', FILTER_SANITIZE_STRING);
            $prov_tipoidfiscal      = filter_input(INPUT_POST, 'tipodoc', FILTER_SANITIZE_NUMBER_INT);
            $prov_cifnif            = filter_input(INPUT_POST, 'dni', FILTER_SANITIZE_STRING);
            $prov_razonsocial       = filter_input(INPUT_POST, 'razon_social', FILTER_SANITIZE_STRING);
            $prov_email             = filter_input(INPUT_POST, 'correo', FILTER_SANITIZE_EMAIL);
            $prov_telefono1         = filter_input(INPUT_POST, 'telefono', FILTER_SANITIZE_STRING);
            $prov_direccion         = filter_input(INPUT_POST, 'direccion_uno', FILTER_SANITIZE_STRING);
            $prov_personafisica     = ('true' == filter_input(INPUT_POST, 'prov_personafisica')) ? 1 : 0;

            if (empty($prov_tipoidfiscal))
                $this->array_errors[] = "El campo cedula de identidad es obligatorio.";
            if (empty($prov_cifnif))
                $this->array_errors[] = "El campo Tipo de documento es obligatorio.";
            if (empty($prov_razonsocial))
                $this->array_errors[] = "El campo Razón social es obligatorio.";
            if (empty($prov_email))
                $this->array_errors[] = "El campo Correo es obligatorio.";
            if (empty($prov_telefono1))
                $this->array_errors[] = "El campo Telefono es obligatorio.";
            if (empty($prov_direccion))
                $this->array_errors[] = "El campo Direccion es obligatorio.";

            if (empty($this->array_errors)) {

                $new_proveedor = new Proveedor();

                if ($new_proveedor->existe('prov_cifnif', $prov_cifnif))
                    $this->array_errors[] = "La cedula de identidad $prov_cifnif ya se encuentra en uso";
                if ($new_proveedor->existe('prov_email', $prov_email))
                    $this->array_errors[] = "El correo electronico $prov_email ya se encuentra en uso";

                //if (valida_email($cli_email) == false)
                //array_push($this->array_errors, "La correo $cli_email no es correcto");
                if (empty($this->array_errors) && empty($this->array_errors)) {
                    $data_id = $new_proveedor->guardar([
                        'prov_tipoidfiscal'     => $prov_tipoidfiscal,
                        'prov_codigo'           => $prov_cifnif,
                        'prov_cifnif'           => $prov_cifnif,
                        'prov_nombre'           => $prov_razonsocial,
                        'prov_razonsocial'      => $prov_razonsocial,
                        'prov_email'            => $prov_email,
                        'prov_telefono1'        => $prov_telefono1,
                        'prov_direccion'        => $prov_direccion,
                        'prov_acreedor'         => 1,
                        'prov_personafisica'    => $prov_personafisica
                    ]);
                    if ($data_id) {
                        $url = ruta_base() . $ruta . "/nuevaFactura/$data_id";
                        Http::json_response($url);
                    } else {
                        Http::json_response(false);
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function obtenerNotasdeCredito()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $anio = filter_input(INPUT_GET, 'anio', FILTER_SANITIZE_NUMBER_INT);
            $mes = $this->solicitud->sanitize($_GET['mes'], FILTER_SANITIZE_NUMBER_INT);

            if ($mes == 0) {
                $activos = $this->compras->obtenerCreditosAnuales("YEAR(cp.crecmp_fecha)=$anio");
                $inactivos = $this->compras->obtenerCreditosAnualesInactivos("YEAR(cp.crecmp_fecha)=$anio");
            } else {
                $activos = $this->compras->obtenerCreditosAnuales("MONTH(cp.crecmp_fecha)=$mes AND YEAR(cp.crecmp_fecha)=$anio");
                $inactivos = $this->compras->obtenerCreditosAnualesInactivos("MONTH(cp.crecmp_fecha)=$mes AND YEAR(cp.crecmp_fecha)=$anio");
            }

            Http::json_response([
                'activos'         => $activos,
                'inactivos'    => $inactivos
            ]);
        }
    }

    public function obtenerNotadeCreditoDetalle($id)
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

            $total = 0;
            $array_pagos = [];
            $pagos = $this->compras->obtenerNotadeCreditoDetalle($id);

            if (count($pagos) > 0) {
                foreach ($pagos as $row) {

                    $divisa_pagada  = $this->compras->obtenerDivisa($row['crecmpart_divid']);
                    $divisa_factura = $this->compras->obtenerDivisa($row['crecmpart_divid_conversion']);

                    $array_pagos[] = [
                        'monto'         => Moneda::moneda($row['crecmpart_monto'], $divisa_pagada['div_locale'], $divisa_pagada['div_simbolo']),
                        'factor'        => $row['crecmpart_factor'],
                        'conversion'    => Moneda::moneda($row['crecmpart_conversion'], $divisa_factura['div_locale'], $divisa_factura['div_simbolo']),
                        'fecha'         => $row['fecha']
                    ];

                    $total += $row['crecmpart_conversion'];
                }

                $total = Moneda::moneda($total, $divisa_factura['div_locale'], $divisa_factura['div_simbolo']);
            }

            Http::json_response([
                'pagos' => $array_pagos,
                'total' => $total
            ]);
        }
    }

    public function finalizarFactura()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if ($this->module->has_module_action_permission('facturacion', 'status', $_SESSION['user_data']['emp_id'])) {

                $id = $this->solicitud->sanitize($_GET['id'], FILTER_SANITIZE_NUMBER_INT);
                $estatus = $this->solicitud->sanitize($_GET['estatus'], FILTER_SANITIZE_NUMBER_INT);

                $newComprasFactura = new Cmpfactura();

                if ($newComprasFactura->editar($id, [
                    'cmp_estatus' => $estatus
                ])) {
                    Http::json_response(true);
                } else {
                    Http::json_response(false);
                }
            } else {
                Http::json_response('Usted no posee permisos para cambiar el estatus de los documentos');
            }
        }
    }

    public function finalizarNota()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if ($this->module->has_module_action_permission('facturacion', 'status', $_SESSION['user_data']['emp_id'])) {

                $id = $this->solicitud->sanitize($_GET['id'], FILTER_SANITIZE_NUMBER_INT);
                $estatus = $this->solicitud->sanitize($_GET['estatus'], FILTER_SANITIZE_NUMBER_INT);

                $newNotaDeEntrega = new Cmpnotas();

                if ($newNotaDeEntrega->editar($id, [
                    'cmpn_estatus' => $estatus
                ])) {
                    Http::json_response(true);
                } else {
                    Http::json_response(false);
                }
            } else {
                Http::json_response('Usted no posee permisos para cambiar el estatus de los documentos');
            }
        }
    }

    public function finalizarPedido()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if ($this->module->has_module_action_permission('facturacion', 'status', $_SESSION['user_data']['emp_id'])) {

                $id = $this->solicitud->sanitize($_GET['id'], FILTER_SANITIZE_NUMBER_INT);
                $estatus = $this->solicitud->sanitize($_GET['estatus'], FILTER_SANITIZE_NUMBER_INT);

                $newNotaDeEntrega = new CmpPedidos();

                if ($newNotaDeEntrega->editar($id, [
                    'cmppe_estatus' => $estatus
                ])) {
                    Http::json_response(true);
                } else {
                    Http::json_response(false);
                }
            } else {
                Http::json_response('Usted no posee permisos para cambiar el estatus de los documentos');
            }
        }
    }

    public function proveedoresReportesAnio()
    {

        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $anio = filter_input(INPUT_POST, 'anio', FILTER_SANITIZE_STRING);

            $artic_total        = 0;
            $costo_total        = 0;
            $venta_total        = 0;
            $totales_notas  = [];
            $total_notas_id = [];
            $ganancia = 0;

            $notas =  $this->compras->obtenerProveedoresNEVAno($anio);
            $factura =  $this->compras->obtenerProveedoresFactAno($anio);
            $orden =  $this->compras->obtenerProveedoresORDAno($anio);

            if (is_countable($notas) && count($notas)) {

                foreach ($notas as $row) {
                    $factor2 = 1;

                    if ($row['ne_divid'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['ne_divid'], $this->div_master['div_id']);
                        $row['neart_procosto'] = round(($row['neart_procosto'] * $factor['df_factor']), 2);
                        $row['neart_prototal'] = round(($row['neart_prototal'] * $factor['df_factor']), 2);
                        $factor2 = $factor['df_factor'];
                    }

                    if ($row['neart_prototal'] > 1) {
                        $ganancia = $row['neart_prototal'] - ($row['neart_procosto'] * $row['neart_procantidad']);
                        $ga = ($ganancia * 100);
                        $ganc = ($row['neart_procosto'] * $row['neart_procantidad']);
                        if ($ga > 0 && $ganc > 0) {
                            $ganancia_porct =  $ga / $ganc;
                        } else {
                            $ganancia_porct = 0;
                        }
                    }

                    if ($row['pro_tipo'] == 2) {
                        $newpro = new ProductosCompuestos();
                        $prod_compuesto = $newpro->obtenerProductosCompuesto($row['pro_id']);

                        foreach ($prod_compuesto as $row2) {
                            $proveed = $this->compras->obtenerProveedores($row2['pro_id']);

                            foreach ($proveed as $row1) {

                                if ($row2['pro_ccprecio'] > 1) {
                                    $ganancia = (($row2['pro_ccprecio'] * $row2['pro_cccant']) - ($row2['pro_costo'] * $row2['pro_cccant']) * $row['neart_procantidad']);
                                    $ga = ($ganancia * 100);
                                    $ganc = ($row2['pro_costo'] * $row2['pro_cccant']) * $row['neart_procantidad'];
                                    if ($ga > 0 && $ganc > 0) {
                                        $ganancia_porct =  $ga / $ganc;
                                    } else {
                                        $ganancia_porct = 0;
                                    }
                                }
                                if (count($totales_notas) < 10) {
                                    if (!in_array($row1['prov_id'], $total_notas_id)) {
                                        $totales_notas[$row1['prov_id']] = [
                                            'div_locale'        => $this->div_master['div_locale'],
                                            'div_simbolo'       => $this->div_master['div_simbolo'],
                                            'div_nombre'        => $this->div_master['div_nombre'],
                                            'proveedor_id'      => $row1['prov_id'],
                                            'productos'         => $row2['pro_descripcion'],
                                            'costo'             => ($row2['pro_costo'] * $row2['pro_cccant']) * $row['neart_procantidad'],
                                            'cantidad'          => $row['neart_procantidad'],
                                            'venta'             => ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['neart_procantidad'],
                                            'ganancia'          => $ganancia,
                                            'sucursal'          => $row['su_nombre'],
                                            'proveedor'         => $row1['prov_razonsocial'],
                                        ];
                                        $total_notas_id[] = $row1['prov_id'];
                                    } else {
                                        $totales_notas[$row1['prov_id']]['costo']       += ($row2['pro_costo'] * $row2['pro_cccant']) * $row['neart_procantidad'];
                                        $totales_notas[$row1['prov_id']]['cantidad']          += $row['neart_procantidad'];
                                        $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                        $totales_notas[$row1['prov_id']]['venta']          += ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['neart_procantidad'];
                                    }

                                    $artic_total +=  $ganancia;
                                    $costo_total +=  ($row2['pro_costo'] * $row2['pro_cccant']) * $row['neart_procantidad'];
                                    $venta_total += ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['neart_procantidad'];
                                }
                            }
                        }
                    } else {
                        $proveed = $this->compras->obtenerProveedores($row['pro_id']);

                        foreach ($proveed as $row1) {
                            if (count($totales_notas) < 10) {
                                if (!in_array($row1['prov_id'], $total_notas_id)) {
                                    $totales_notas[$row1['prov_id']] = [
                                        'div_locale'        => $this->div_master['div_locale'],
                                        'div_simbolo'       => $this->div_master['div_simbolo'],
                                        'div_nombre'        => $this->div_master['div_nombre'],
                                        'proveedor_id'      => $row1['prov_id'],
                                        'productos'         => $row['pro_descripcion'],
                                        'costo'             => $row['neart_procosto'] * $row['neart_procantidad'],
                                        'cantidad'          => $row['neart_procantidad'],
                                        'venta'             => $row['neart_prototal'],
                                        'ganancia'          => $ganancia,
                                        'sucursal'          => $row['su_nombre'],
                                        'proveedor'         => $row1['prov_razonsocial'],
                                    ];
                                    $total_notas_id[] = $row1['prov_id'];
                                } else {
                                    $totales_notas[$row1['prov_id']]['costo']       +=  $row['neart_procosto'] * $row['neart_procantidad'];
                                    $totales_notas[$row1['prov_id']]['cantidad']          += $row['neart_procantidad'];
                                    $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                    $totales_notas[$row1['prov_id']]['venta']          +=  $row['neart_prototal'];
                                }

                                $artic_total +=  $ganancia;
                                $costo_total +=  $row['neart_procosto'] * $row['neart_procantidad'];
                                $venta_total += $row['neart_prototal'];
                            }
                        }
                    }
                }
            }

            if (is_countable($factura) && count($factura)) {

                foreach ($factura as $row) {
                    $factor2 = 1;

                    if ($row['fm_divid'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['fm_divid'], $this->div_master['div_id']);
                        $row['fa_procosto'] = round(($row['fa_procosto'] * $factor['df_factor']), 2);
                        $row['fa_prototal'] = round(($row['fa_prototal'] * $factor['df_factor']), 2);
                        $factor2 = $factor['df_factor'];
                    }

                    if ($row['fa_prototal'] > 1) {
                        $ganancia = $row['fa_prototal'] - ($row['fa_procosto'] * $row['fa_procantidad']);
                        $ga = ($ganancia * 100);
                        $ganc = ($row['fa_procosto'] * $row['fa_procantidad']);
                        if ($ga > 0 && $ganc > 0) {
                            $ganancia_porct =  $ga / $ganc;
                        } else {
                            $ganancia_porct = 0;
                        }
                    }

                    if ($row['pro_tipo'] == 2) {
                        $newpro = new ProductosCompuestos();
                        $prod_compuesto = $newpro->obtenerProductosCompuesto($row['pro_id']);

                        foreach ($prod_compuesto as $row2) {
                            $proveed = $this->compras->obtenerProveedores($row['pro_id']);

                            if ($row['fa_prototal'] > 1) {
                                $ganancia = (($row2['pro_ccprecio'] * $row2['pro_cccant']) - ($row2['pro_costo'] * $row2['pro_cccant']) * $row['fa_procantidad']);
                                $ga = ($ganancia * 100);
                                $ganc = ($row2['pro_costo'] * $row2['pro_cccant']) * $row['fa_procantidad'];
                                if ($ga > 0 && $ganc > 0) {
                                    $ganancia_porct =  $ga / $ganc;
                                } else {
                                    $ganancia_porct = 0;
                                }
                            }

                            foreach ($proveed as $row1) {
                                if (count($totales_notas) < 10) {
                                    if (!in_array($row1['prov_id'], $total_notas_id)) {
                                        $totales_notas[$row1['prov_id']] = [
                                            'div_locale'        => $this->div_master['div_locale'],
                                            'div_simbolo'       => $this->div_master['div_simbolo'],
                                            'div_nombre'        => $this->div_master['div_nombre'],
                                            'proveedor_id'      => $row1['prov_id'],
                                            'productos'         => $row['pro_descripcion'],
                                            'costo'             => ($row2['pro_costo'] * $row2['pro_cccant']) * $row['fa_procantidad'],
                                            'cantidad'          => $row['fa_procantidad'],
                                            'venta'             => ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['fa_procantidad'],
                                            'ganancia'          => $ganancia,
                                            'sucursal'          => $row['su_nombre'],
                                            'proveedor'         => $row1['prov_razonsocial'],
                                        ];
                                        $total_notas_id[] = $row1['prov_id'];
                                    } else {
                                        $totales_notas[$row1['prov_id']]['costo']       += ($row2['pro_costo'] * $row2['pro_cccant']) * $row['fa_procantidad'];
                                        $totales_notas[$row1['prov_id']]['cantidad']          += $row['fa_procantidad'];
                                        $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                        $totales_notas[$row1['prov_id']]['venta']          +=  ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['fa_procantidad'];
                                    }

                                    $artic_total +=  $ganancia;
                                    $costo_total +=  ($row2['pro_costo'] * $row2['pro_cccant']) * $row['fa_procantidad'];
                                    $venta_total += ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['fa_procantidad'];
                                }
                            }
                        }
                    } else {
                        $proveed = $this->compras->obtenerProveedores($row['pro_id']);

                        foreach ($proveed as $row1) {
                            if (count($totales_notas) < 10) {
                                if (!in_array($row1['prov_id'], $total_notas_id)) {
                                    $totales_notas[$row1['prov_id']] = [
                                        'div_locale'        => $this->div_master['div_locale'],
                                        'div_simbolo'       => $this->div_master['div_simbolo'],
                                        'div_nombre'        => $this->div_master['div_nombre'],
                                        'proveedor_id'      => $row1['prov_id'],
                                        'productos'         => $row['pro_descripcion'],
                                        'costo'             => $row['fa_procosto'] * $row['fa_procantidad'],
                                        'cantidad'          => $row['fa_procantidad'],
                                        'venta'             => $row['fa_prototal'],
                                        'ganancia'          => $ganancia,
                                        'sucursal'          => $row['su_nombre'],
                                        'proveedor'         => $row1['prov_razonsocial'],
                                    ];
                                    $total_notas_id[] = $row1['prov_id'];
                                } else {
                                    $totales_notas[$row1['prov_id']]['costo']       +=  $row['fa_procosto'] * $row['fa_procantidad'];
                                    $totales_notas[$row1['prov_id']]['cantidad']          += $row['fa_procantidad'];
                                    $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                    $totales_notas[$row1['prov_id']]['venta']          +=  $row['fa_prototal'];
                                }

                                $artic_total +=  $ganancia;
                                $costo_total +=  $row['fa_procosto'] * $row['fa_procantidad'];
                                $venta_total += $row['fa_prototal'];
                            }
                        }
                    }
                }
            }

            if (is_countable($orden) && count($orden)) {

                foreach ($orden as $row) {
                    $factor2 = 1;

                    if ($row['n_divid'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['n_divid'], $this->div_master['div_id']);
                        $row['nart_procosto'] = round(($row['nart_procosto'] * $factor['df_factor']), 2);
                        $row['nart_prototal'] = round(($row['nart_prototal'] * $factor['df_factor']), 2);
                        $factor2 = $factor['df_factor'];
                    }

                    if ($row['nart_prototal'] > 1) {
                        $ganancia = $row['nart_prototal'] - ($row['nart_procosto'] * $row['nart_procantidad']);
                        $ga = ($ganancia * 100);
                        $ganc = ($row['nart_procosto'] * $row['nart_procantidad']);
                        if ($ga > 0 && $ganc > 0) {
                            $ganancia_porct =  $ga / $ganc;
                        } else {
                            $ganancia_porct = 0;
                        }
                    }

                    if ($row['pro_tipo'] == 2) {
                        $newpro = new ProductosCompuestos();
                        $prod_compuesto = $newpro->obtenerProductosCompuesto($row['pro_id']);

                        foreach ($prod_compuesto as $row2) {
                            $proveed = $this->compras->obtenerProveedores($row2['pro_id']);

                            foreach ($proveed as $row1) {

                                if ($row2['pro_ccprecio'] > 1) {
                                    $ganancia = (($row2['pro_ccprecio'] * $row2['pro_cccant']) - ($row2['pro_costo'] * $row2['pro_cccant']) * $row['neart_procantidad']);
                                    $ga = ($ganancia * 100);
                                    $ganc = ($row2['pro_costo'] * $row2['pro_cccant']) * $row['nart_procantidad'];
                                    if ($ga > 0 && $ganc > 0) {
                                        $ganancia_porct =  $ga / $ganc;
                                    } else {
                                        $ganancia_porct = 0;
                                    }
                                }
                                if (count($totales_notas) < 10) {
                                    if (!in_array($row1['prov_id'], $total_notas_id)) {
                                        $totales_notas[$row1['prov_id']] = [
                                            'div_locale'        => $this->div_master['div_locale'],
                                            'div_simbolo'       => $this->div_master['div_simbolo'],
                                            'div_nombre'        => $this->div_master['div_nombre'],
                                            'proveedor_id'      => $row1['prov_id'],
                                            'productos'         => $row2['pro_descripcion'],
                                            'costo'             => ($row2['pro_costo'] * $row2['pro_cccant']) * $row['nart_procantidad'],
                                            'cantidad'          => $row['nart_procantidad'],
                                            'venta'             => ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['nart_procantidad'],
                                            'ganancia'          => $ganancia,
                                            'sucursal'          => $row['su_nombre'],
                                            'proveedor'         => $row1['prov_razonsocial'],
                                        ];
                                        $total_notas_id[] = $row1['prov_id'];
                                    } else {
                                        $totales_notas[$row1['prov_id']]['costo']       += ($row2['pro_costo'] * $row2['pro_cccant']) * $row['nart_procantidad'];
                                        $totales_notas[$row1['prov_id']]['cantidad']          += $row['nart_procantidad'];
                                        $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                        $totales_notas[$row1['prov_id']]['venta']          += ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['nart_procantidad'];
                                    }
                                }
                                $artic_total +=  $ganancia;
                                $costo_total +=  ($row2['pro_costo'] * $row2['pro_cccant']) * $row['nart_procantidad'];
                                $venta_total += ($row2['pro_ccprecio'] * $row2['pro_cccant']) * $row['nart_procantidad'];
                            }
                        }
                    } else {
                        $proveed = $this->compras->obtenerProveedores($row['pro_id']);

                        foreach ($proveed as $row1) {
                            if (count($totales_notas) < 10) {
                                if (!in_array($row1['prov_id'], $total_notas_id)) {
                                    $totales_notas[$row1['prov_id']] = [
                                        'div_locale'        => $this->div_master['div_locale'],
                                        'div_simbolo'       => $this->div_master['div_simbolo'],
                                        'div_nombre'        => $this->div_master['div_nombre'],
                                        'proveedor_id'      => $row1['prov_id'],
                                        'productos'         => $row['pro_descripcion'],
                                        'costo'             => $row['nart_procosto'] * $row['nart_procantidad'],
                                        'cantidad'          => $row['nart_procantidad'],
                                        'venta'             => $row['nart_prototal'],
                                        'ganancia'          => $ganancia,
                                        'sucursal'          => $row['su_nombre'],
                                        'proveedor'         => $row1['prov_razonsocial'],
                                    ];
                                    $total_notas_id[] = $row1['prov_id'];
                                } else {
                                    $totales_notas[$row1['prov_id']]['costo']       +=  $row['neart_procosto'] * $row['nart_procantidad'];
                                    $totales_notas[$row1['prov_id']]['cantidad']          += $row['nart_procantidad'];
                                    $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                    $totales_notas[$row1['prov_id']]['venta']          +=  $row['nart_prototal'];
                                }

                                $artic_total +=  $ganancia;
                                $costo_total +=  $row['nart_procosto'] * $row['nart_procantidad'];
                                $venta_total += $row['nart_prototal'];
                            }
                        }
                    }
                }
            }

            /* if (is_countable($notas) && count($notas)) {
                foreach ($notas as $row) {
                    $factor2 = 1;

                    if ($row['ne_divid'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['ne_divid'], $this->div_master['div_id']);
                        $row['neart_procosto'] = round(($row['neart_procosto'] * $factor['df_factor']), 2);
                        $row['neart_prototal'] = round(($row['neart_prototal'] * $factor['df_factor']), 2);
                        $factor2 = $factor['df_factor'];
                    }

                    if ($row['neart_prototal'] > 1) {
                        $ganancia = $row['neart_prototal'] - ($row['neart_procosto'] * $row['neart_procantidad']);
                        $ga = ($ganancia * 100);
                        $ganc = ($row['neart_procosto'] * $row['neart_procantidad']);
                        if ($ga > 0 && $ganc > 0) {
                            $ganancia_porct =  $ga / $ganc;
                        } else {
                            $ganancia_porct = 0;
                        }
                    }

                    $proveed = $this->compras->obtenerProveedores($row['pro_id']);

                    foreach ($proveed as $row1) {
                        if (count($totales_notas) < 10) {
                            if (!in_array($row1['prov_id'], $total_notas_id)) {
                                $totales_notas[$row1['prov_id']] = [
                                    'div_locale'        => $this->div_master['div_locale'],
                                    'div_simbolo'       => $this->div_master['div_simbolo'],
                                    'div_nombre'        => $this->div_master['div_nombre'],
                                    'proveedor_id'      => $row1['prov_id'],
                                    'costo'             => $row['neart_procosto'] * $row['neart_procantidad'],
                                    'cantidad'          => $row['neart_procantidad'],
                                    'venta'             => $row['neart_prototal'],
                                    'ganancia'          => $ganancia,
                                    'sucursal'          => $row['su_nombre'],
                                    'proveedor'         => $row1['prov_razonsocial'],
                                ];
                                $total_notas_id[] = $row1['prov_id'];
                            } else {
                                $totales_notas[$row1['prov_id']]['costo']       +=  $row['neart_procosto'] * $row['neart_procantidad'];
                                $totales_notas[$row1['prov_id']]['cantidad']          += $row['neart_procantidad'];
                                $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                $totales_notas[$row1['prov_id']]['venta']          +=  $row['neart_prototal'];
                            }

                            $artic_total += $ganancia;
                            $costo_total +=  $row['neart_procosto'] * $row['neart_procantidad'];
                            $venta_total += $row['neart_prototal'];
                        }
                    }
                }
            }
            if (is_countable($factura) && count($factura)) {
                foreach ($factura as $row) {
                    $factor2 = 1;

                    if ($row['fm_divid'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['fm_divid'], $this->div_master['div_id']);
                        $row['fa_procosto'] = round(($row['fa_procosto'] * $factor['df_factor']), 2);
                        $row['fa_prototal'] = round(($row['fa_prototal'] * $factor['df_factor']), 2);
                        $factor2 = $factor['df_factor'];
                    }

                    if ($row['fa_prototal'] > 1) {
                        $ganancia = $row['fa_prototal'] - ($row['fa_procosto'] * $row['fa_procantidad']);
                        $ga = ($ganancia * 100);
                        $ganc = ($row['fa_procosto'] * $row['fa_procantidad']);
                        if ($ga > 0 && $ganc > 0) {
                            $ganancia_porct =  $ga / $ganc;
                        } else {
                            $ganancia_porct = 0;
                        }
                    }

                    $proveed = $this->compras->obtenerProveedores($row['pro_id']);

                    foreach ($proveed as $row1) {
                        if (count($totales_notas) < 10) {
                            if (!in_array($row1['prov_id'], $total_notas_id)) {
                                $totales_notas[$row1['prov_id']] = [
                                    'div_locale'        => $this->div_master['div_locale'],
                                    'div_simbolo'       => $this->div_master['div_simbolo'],
                                    'div_nombre'        => $this->div_master['div_nombre'],
                                    'proveedor_id'      => $row1['prov_id'],
                                    'costo'             => $row['fa_procosto'] * $row['fa_procantidad'],
                                    'cantidad'          => $row['fa_procantidad'],
                                    'venta'             => $row['fa_prototal'],
                                    'ganancia'          => $ganancia,
                                    'sucursal'          => $row['su_nombre'],
                                    'proveedor'         => $row1['prov_razonsocial'],
                                ];
                                $total_notas_id[] = $row1['prov_id'];
                            } else {
                                $totales_notas[$row1['prov_id']]['costo']       +=  $row['fa_procosto'] * $row['fa_procantidad'];
                                $totales_notas[$row1['prov_id']]['cantidad']          += $row['fa_procantidad'];
                                $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                $totales_notas[$row1['prov_id']]['venta']          +=  $row['fa_prototal'];
                            }

                            $artic_total += $ganancia;
                            $costo_total +=  $row['fa_procosto'] * $row['fa_procantidad'];
                            $venta_total += $row['fa_prototal'];
                        }
                    }
                }
            }
            if (is_countable($orden) && count($orden)) {
                foreach ($orden as $row) {
                    $factor2 = 1;

                    if ($row['n_divid'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['n_divid'], $this->div_master['div_id']);
                        $row['nart_procosto'] = round(($row['nart_procosto'] * $factor['df_factor']), 2);
                        $row['nart_prototal'] = round(($row['nart_prototal'] * $factor['df_factor']), 2);
                        $factor2 = $factor['df_factor'];
                    }

                    if ($row['nart_prototal'] > 1) {
                        $ganancia = $row['nart_prototal'] - ($row['nart_procosto'] * $row['nart_procantidad']);
                        $ga = ($ganancia * 100);
                        $ganc = ($row['nart_procosto'] * $row['nart_procantidad']);
                        if ($ga > 0 && $ganc > 0) {
                            $ganancia_porct =  $ga / $ganc;
                        } else {
                            $ganancia_porct = 0;
                        }
                    }

                    $proveed = $this->compras->obtenerProveedores($row['pro_id']);

                    foreach ($proveed as $row1) {
                        if (count($totales_notas) < 10) {
                            if (!in_array($row1['prov_id'], $total_notas_id)) {
                                $totales_notas[$row1['prov_id']] = [
                                    'div_locale'        => $this->div_master['div_locale'],
                                    'div_simbolo'       => $this->div_master['div_simbolo'],
                                    'div_nombre'        => $this->div_master['div_nombre'],
                                    'proveedor_id'      => $row1['prov_id'],
                                    'costo'             => $row['nart_procosto'] * $row['nart_procantidad'],
                                    'cantidad'          => $row['nart_procantidad'],
                                    'venta'             => $row['nart_prototal'],
                                    'ganancia'          => $ganancia,
                                    'sucursal'          => $row['su_nombre'],
                                    'proveedor'         => $row1['prov_razonsocial'],
                                ];
                                $total_notas_id[] = $row1['prov_id'];
                            } else {
                                $totales_notas[$row1['prov_id']]['costo']       +=  $row['nart_procosto'] * $row['nart_procantidad'];
                                $totales_notas[$row1['prov_id']]['cantidad']          += $row['nart_procantidad'];
                                $totales_notas[$row1['prov_id']]['ganancia']          += $ganancia;
                                $totales_notas[$row1['prov_id']]['venta']          +=  $row['nart_prototal'];
                            }

                            $artic_total += $ganancia;
                            $costo_total +=  $row['neart_procosto'] * $row['nart_procantidad'];
                            $venta_total += $row['neart_prototal'];
                        }
                    }
                }
            }*/

            Http::json_response([
                'proveedores'     => $totales_notas,
                'ganancia' => $artic_total,
                'costo' => $costo_total,
                'venta' => $venta_total,
                'div_locale'        => $this->div_master['div_locale'],
                'div_simbolo'       => $this->div_master['div_simbolo'],
                'div_nombre'        => $this->div_master['div_nombre'],
            ]);
        }
    }

    public function searc($valor, $array)
    {
        foreach ($array as $key => $val) {
            if ($val['id'] === $valor) {
                return $key;
            }
        }
        return 'vacio';
    }

    public function generarExcel()
    {

        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $mes  = filter_input(INPUT_POST, 'mes', FILTER_SANITIZE_STRING);
            $anio  = filter_input(INPUT_POST, 'anio', FILTER_SANITIZE_STRING);
            $divisa  = filter_input(INPUT_POST, 'divisa', FILTER_SANITIZE_STRING);

            $productos = json_decode($_POST['productos'], true);
            $product = new Productos();
            $empresa_data       = $product->obtenerDatosEmpresa();

            // Http::json_response($productos);
            $array_productos    = $productos;


            $spreadsheet = new Spreadsheet();

            $spreadsheet->getProperties()->setCreator($empresa_data['emp_nombre'])->setLastModifiedBy('Firma Digital')->setTitle('Listado de productos')->setSubject('Office 2007 XLSX Test Document')->setDescription('Test document for Office 2007 XLSX, generated using PHP classes.')->setKeywords('producto tarifa office 2007 openxml php')->setCategory('Listado de productos');

            $spreadsheet->getActiveSheet()
                ->setTitle('REPORTE')
                ->setCellValue('A1', 'REGISTRO DETALLADO DE ENTRADAS Y SALIDAS DE INVENTARIOS');
            $spreadsheet->getActiveSheet()->getRowDimension('1')->setRowHeight(40);
            $spreadsheet->getActiveSheet()->mergeCells('A1:O1');
            $spreadsheet->getActiveSheet()->getStyle('A1:O1')->getFont()->setBold(true)->setSize(16);
            $spreadsheet->getActiveSheet()->getStyle('A1:O1')->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setRGB('9FD5D1');
            $spreadsheet->getActiveSheet()->getStyle('A1:O1')->getAlignment()->setHorizontal('center');

            $spreadsheet->getActiveSheet()->getStyle('D3')->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getCell('D3')->setValue('MES: ');
            $spreadsheet->getActiveSheet()->getStyle('D3')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getStyle('E3')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getCell('E3')->setValue($mes);

            $spreadsheet->getActiveSheet()->getStyle('D4')->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getCell('D4')->setValue('EJERCICIO GRAVABLE: ');
            $spreadsheet->getActiveSheet()->getStyle('D4')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getStyle('E4')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getCell('E4')->setValue($anio);

            /*$spreadsheet->getActiveSheet()->getStyle('D5')->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getCell('D5')->setValue('NOMBRE DE LA PERSONA NATURAL: ');
            $spreadsheet->getActiveSheet()->getStyle('D5')->getAlignment()->setHorizontal('right');
            $spreadsheet->getActiveSheet()->getCell('E5')->setValue('JUNIO');*/

            $spreadsheet->getActiveSheet()->getStyle('D5')->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getCell('D5')->setValue('RAZON SOCIAL: ');
            $spreadsheet->getActiveSheet()->getStyle('D5')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getStyle('E5')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getCell('E5')->setValue($empresa_data['emp_nombre']);

            $spreadsheet->getActiveSheet()->getStyle('D6')->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getCell('D6')->setValue('RIF DEL CONTRIBUYENTE: ');
            $spreadsheet->getActiveSheet()->getStyle('D6')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getStyle('E6')->getAlignment()->setHorizontal('left');
            $spreadsheet->getActiveSheet()->getCell('E6')->setValue($empresa_data['emp_rif']);

            $spreadsheet->getActiveSheet()->setCellValue("D8", "UNIDADES");
            $spreadsheet->getActiveSheet()->mergeCells('D8:I8');
            $spreadsheet->getActiveSheet()->getStyle('D8:I8')->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getStyle('D8:I8')->getBorders()->getOutline()->setBorderStyle(StyleBorder::BORDER_DASHDOT)->setColor(new Color('9FD5D1'));
            $spreadsheet->getActiveSheet()->getStyle('D8:I8')->getAlignment()->setHorizontal('center');

            $spreadsheet->getActiveSheet()->setCellValue("J8", $divisa);
            $spreadsheet->getActiveSheet()->mergeCells('J8:O8');
            $spreadsheet->getActiveSheet()->getStyle('J8:O8')->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getStyle('J8:O8')->getBorders()->getOutline()->setBorderStyle(StyleBorder::BORDER_DASHDOT)->setColor(new Color('9FD5D1'));
            $spreadsheet->getActiveSheet()->getStyle('J8:O8')->getAlignment()->setHorizontal('center');

            $spreadsheet->getActiveSheet()->getRowDimension('9')->setRowHeight(40);
            $spreadsheet->getActiveSheet()->getColumnDimension('A')->setWidth(100);
            $spreadsheet->getActiveSheet()->getStyle("A9:O9")->getBorders()->getAllBorders()->setBorderStyle(StyleBorder::BORDER_MEDIUM)->setColor(new Color('9FD5D1'));
            $spreadsheet->getActiveSheet()->getStyle("A9:O9")->getAlignment()->setHorizontal('right');
            $spreadsheet->getActiveSheet(0)->setCellValue("A9", 'N-')
                ->setCellValue('B9', 'ITEM')
                ->setCellValue('C9', 'DESCRIPCION')
                ->setCellValue('D9', 'EXIST. ANT')
                ->setCellValue('E9', 'ENTRADAS')
                ->setCellValue('F9', 'SALIDAS')
                ->setCellValue('G9', 'RETIROS')
                ->setCellValue('H9', 'AUTOCONSUMOS')
                ->setCellValue('I9', 'EXISTENCIA')
                ->setCellValue('J9', 'VALOR ANTERIOR EN BS')
                ->setCellValue('K9', 'ENTRADAS')
                ->setCellValue('L9', 'SALIDAS')
                ->setCellValue('M9', 'RETIROS')
                ->setCellValue('N9', 'AUTOCONSUMOS')
                ->setCellValue('O9', 'EXISTENCIA');
            $spreadsheet->getActiveSheet()->getColumnDimension('A')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('B')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('C')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('D')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('E')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('F')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('G')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('H')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('I')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('J')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('K')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('L')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('M')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('N')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('O')->setAutoSize(true);

            $i = 10;
            $k = 1;
            $cantid1 = 0;
            $cantid2 = 0;
            $cantid3 = 0;
            $cantid4 = 0;
            $cantid5 = 0;
            $cantid6 = 0;
            $precio1 = 0;
            $precio2 = 0;
            $precio3 = 0;
            $precio4 = 0;
            $precio5 = 0;
            $precio6 = 0;

            foreach ($array_productos as $pro) {

                //$spreadsheet->getActiveSheet()->setCellValue("A$i", $row);
                $anterior = 0;
                if (count($pro['facturas']) > 0) {
                    foreach ($pro['facturas'] as $fact) {
                        $anterior += $fact['cantidad'];
                    }

                    $existep = $pro['precio'] * ($pro['cantidad'] + $anterior);
                    $spreadsheet->getActiveSheet()->getRowDimension("$i")->setRowHeight(20);
                    $spreadsheet->getActiveSheet()->getColumnDimension('A')->setWidth(100);
                    $spreadsheet->getActiveSheet()->getStyle("A$i:O$i")->getBorders()->getAllBorders()->setBorderStyle(StyleBorder::BORDER_MEDIUM)->setColor(new Color('9FD5D1'));

                    $spreadsheet->getActiveSheet()
                        ->setCellValue("A$i", $k)
                        ->setCellValue("B$i", $pro['codigo'])
                        ->setCellValue("C$i", $pro['producto'])
                        ->setCellValue("D$i", $pro['cantidad'] + $anterior)
                        ->setCellValue("E$i", $pro['cantidad'] + $anterior)
                        ->setCellValue("F$i", $pro['cantidad'])
                        ->setCellValue("G$i", 0)
                        ->setCellValue("H$i", 0)
                        ->setCellValue("I$i", $anterior)
                        ->setCellValue("J$i", round($existep, 2))
                        ->setCellValue("K$i", round($existep, 2))
                        ->setCellValue("L$i", round($pro['precio'] * $pro['cantidad'], 2))
                        ->setCellValue("M$i", 0)
                        ->setCellValue("N$i", 0)
                        ->setCellValue("O$i", round($pro['precio'] * $anterior, 2));

                    $cantid1 += $pro['cantidad'] + $anterior;
                    $cantid2 += $pro['cantidad'] + $anterior;
                    $cantid3 += $pro['cantidad'];
                    $cantid4 = 0;
                    $cantid5 = 0;
                    $cantid6 += $anterior;
                    $precio1 += round($existep, 2);
                    $precio2 += round($existep, 2);
                    $precio3 += round($pro['precio'] * $pro['cantidad'], 2);
                    $precio4 = 0;
                    $precio5 = 0;
                    $precio6 += round($pro['precio'] * $anterior, 2);
                }else{
                    $spreadsheet->getActiveSheet()->getRowDimension("$i")->setRowHeight(20);
                    $spreadsheet->getActiveSheet()->getColumnDimension('A')->setWidth(100);
                    $spreadsheet->getActiveSheet()->getStyle("A$i:O$i")->getBorders()->getAllBorders()->setBorderStyle(StyleBorder::BORDER_MEDIUM)->setColor(new Color('9FD5D1'));

                    $spreadsheet->getActiveSheet()
                    ->setCellValue("A$i", $k)
                    ->setCellValue("B$i", $pro['codigo'])
                    ->setCellValue("C$i", $pro['producto'])
                    ->setCellValue("D$i", 0)
                    ->setCellValue("E$i", 0)
                    ->setCellValue("F$i", $pro['cantidad'])
                    ->setCellValue("G$i", 0)
                    ->setCellValue("H$i", 0)
                    ->setCellValue("I$i", 0)
                    ->setCellValue("J$i",0)
                    ->setCellValue("K$i", 0)
                    ->setCellValue("L$i", round($pro['precio'] * $pro['cantidad'], 2))
                    ->setCellValue("M$i", 0)
                    ->setCellValue("N$i", 0)
                    ->setCellValue("O$i", 0);

                    $cantid3+= $pro['cantidad'];
                    $precio3+=round($pro['precio'] * $pro['cantidad'], 2);
                }

                $i++;
                $k++;
            }

            $spreadsheet->getActiveSheet()->getStyle("C$i:O$i")->getFont()->setBold(true)->setSize(12);
            $spreadsheet->getActiveSheet()->getStyle("D$i:O$i")->getBorders()->getBottom()->setBorderStyle(StyleBorder::BORDER_DASHDOT)->setColor(new Color('9FD5D1'));
            $spreadsheet->getActiveSheet()->getStyle("D$i:O$i")->getAlignment()->setHorizontal('right');

            $spreadsheet->getActiveSheet()->getRowDimension('1')->setRowHeight(20);
            $spreadsheet->getActiveSheet()->getColumnDimension('A')->setWidth(100);
            $spreadsheet->getActiveSheet()
                ->setCellValue("A$i", "")
                ->setCellValue("B$i", "")
                ->setCellValue("C$i", "TOTALES")
                ->setCellValue("D$i", $cantid1)
                ->setCellValue("E$i", $cantid2)
                ->setCellValue("F$i", $cantid3)
                ->setCellValue("G$i", 0)
                ->setCellValue("H$i", 0)
                ->setCellValue("I$i",  $cantid6)
                ->setCellValue("J$i",  round($precio1, 2))
                ->setCellValue("K$i",  round($precio2, 2))
                ->setCellValue("L$i",  round($precio3, 2))
                ->setCellValue("M$i", 0)
                ->setCellValue("N$i", 0)
                ->setCellValue("O$i", round($precio6, 2));
            $spreadsheet->getActiveSheet()->getColumnDimension('A')->setAutoSize(true);
            $spreadsheet->getActiveSheet()->getColumnDimension('B')->setAutoSize(true);

            $spreadsheet->getActiveSheet()->setTitle('Listado de productos');
            //$spreadsheet->getActiveSheet()->getProtection()->setSheet(true);

            // Set active sheet index to the first sheet, so Excel opens this as the first sheet
            $spreadsheet->setActiveSheetIndex(0);

            // Redirect output to a client’s web browser (Xlsx)
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="Reporte de inventario.xlsx"');
            header('Cache-Control: max-age=0');
            // If you're serving to IE 9, then the following may be needed
            header('Cache-Control: max-age=1');

            // If you're serving to IE over SSL, then the following may be needed
            header('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); // Date in the past
            header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT'); // always modified
            header('Cache-Control: cache, must-revalidate'); // HTTP/1.1
            header('Pragma: public'); // HTTP/1.0

            $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
            $writer->save('php://output');
            exit;
        }
    }

    public function productosVendidosComprasMes()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $anio   = filter_input(INPUT_GET, 'anio', FILTER_SANITIZE_NUMBER_INT);
            $div_id = filter_input(INPUT_GET, 'divisa', FILTER_SANITIZE_NUMBER_INT);
            $sucursal = filter_input(INPUT_GET, 'sucursal', FILTER_SANITIZE_NUMBER_INT);

            if (empty($anio))
                $anio = date('Y');

            if (empty($this->array_errors)) {

                $array_notas        = [];
                $array_notas_id     = [];
                $artic_total        = 0;
                $costo              = 0;
                $venta              = 0;
                $ganancia           = 0;
                $porcentaje         = 0;
                $array_notas_proid  = [];
                $array_compras_proidNo = [];
                $array_compra_proidDisp  = [];

                $productos =  $this->ingresosegresos_model->obtenerProductossinServicios();

                if (is_countable($productos) && count($productos)) {

                    foreach ($productos as $pro) {
                        $array_compra_proidmes = [];

                        $facturas = $this->ingresosegresos_model->obtenerProduccionFACVMesCompras($anio, $div_id, $sucursal, $pro['pro_id']);
                        $divisa = $this->ingresosegresos_model->obtenerDivisa($div_id);
                        $compras = $this->ingresosegresos_model->obtenerFacturasCompras($pro['pro_id'], $anio);

                        $precio = round($pro['pro_precio'], 2);

                        if ($this->div_master['div_id'] != $div_id) {
                            $div_factor = $this->ingresosegresos_model->divisaFactor($this->div_master['div_id'], $div_id);
                            $precio = round($pro['pro_precio'] * $div_factor['df_factor'], 2);
                        }

                        if (is_countable($facturas) && count($facturas)) {

                            foreach ($facturas as $row) {

                                if ((!in_array($row['mes'], $array_notas_id))) {

                                    if ($row['total_venta'] > 0) {

                                        if ($this->div_master['div_id'] != $div_id) {
                                            $div_factor = $this->ingresosegresos_model->divisaFactor($this->div_master['div_id'], $div_id);
                                            $row['total_costo'] *= $div_factor['df_factor'];
                                            $precio = $pro['pro_precio'] * $div_factor['df_factor'];
                                        }

                                        $porc = round((($row['total_venta'] - $row['total_costo']) * 100 / $row['total_venta']), 2);
                                    } else {
                                        $porc = 0;
                                    }

                                    foreach ($compras as $row1) {

                                        if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) === 'vacio') && ($this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo) === 'vacio')) {
                                            $cant = convertir_a_float($row1['cmpart_procantidad']);

                                            if ($cant > 0) {
                                                if ($cant == convertir_a_float($row['cantidad'])) {
                                                    $array_compras_proidNo[] = [
                                                        'id'       => $row1['codigo'] . $row['producto'],
                                                        'activo'   => 'inactivo',
                                                        'cantidad' => $cant,
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];
                                                    $array_compra_proidmes[] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                        'cantidad' => 0
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => 0,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                } else {
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if ($r > 0) {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'activo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    } else {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'inactivo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    }

                                                    $array_compra_proidmes[] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                        'cantidad' => ($r > 0) ? $r : 0
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => $r,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                }
                                            }
                                        } else {
                                            if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) >= 0)) {
                                                $key = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp);
                                                $cant = convertir_a_float($array_compra_proidDisp[$key]['cantidad']);

                                                if ($cant > 0) {
                                                    $key2 = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo);
                                                    $act = $array_compras_proidNo[$key2]['activo'];
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if (($r > 0) && ($act === 'activo')) {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'activo';
                                                    } else {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'inactivo';
                                                    }
                                                } else {
                                                    $array_compra_proidmes = [];
                                                }
                                            }
                                        }
                                    }

                                    $dta = [];
                                    $dta[$row['producto']] = [
                                        'cantidad'      => $row['cantidad'],
                                        'div_locale'    => $divisa['div_locale'],
                                        'div_simbolo'   => $divisa['div_simbolo'],
                                        'mes'           => $row['mes'],
                                        'costo'         => $row['total_costo'],
                                        'venta'         => $row['total_venta'],
                                        'ganancia'      => $row['total_venta'] - $row['total_costo'],
                                        'porcentaje'    => $porc,
                                        'id'            => $pro['pro_id'],
                                        'precio'        => $precio,
                                        'codigo'        => $pro['pro_codigo'],
                                        'producto'      => $pro['pro_descripcion'] . '-' . $pro['pro_nombrecorto'],
                                        'sucursal'      => $row['su_nombre'],
                                        'facturas'      => $array_compra_proidmes
                                    ];

                                    // $data[$row['producto']] = $dta;

                                    //if ((!in_array($row['mes'] . $row['producto'], $array_notas_proid))) {
                                    $array_notas[$row['mes']] = [
                                        'productos'     => $dta,
                                        'mes'           => $row['mes'],
                                        'divisa'        => $divisa['div_descripcion'],
                                        'anio'          => $anio
                                    ];

                                    $array_notas_id[] = $row['mes'];
                                    $array_notas_proid[] = $row['mes'] . $row['producto'];
                                    //}
                                } else {

                                    if ($row['total_venta'] > 0) {

                                        if ($this->div_master['div_id'] != $div_id) {
                                            $div_factor = $this->ingresosegresos_model->divisaFactor($this->div_master['div_id'], $div_id);
                                            $row['total_costo'] *= $div_factor['df_factor'];
                                        }

                                        $porc = round((($row['total_venta'] - $row['total_costo']) * 100 / $row['total_venta']), 2);
                                    } else {
                                        $porc = 0;
                                    }

                                    foreach ($compras as $row1) {

                                        if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) === 'vacio') && ($this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo) === 'vacio')) {
                                            $cant = convertir_a_float($row1['cmpart_procantidad']);

                                            if ($cant > 0) {
                                                if ($cant == convertir_a_float($row['cantidad'])) {
                                                    $array_compras_proidNo[] = [
                                                        'id'       => $row1['codigo'] . $row['producto'],
                                                        'activo'   => 'inactivo',
                                                        'cantidad' => $cant,
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];
                                                    $array_compra_proidmes[] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                        'cantidad' => 0
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => 0,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                } else {
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if ($r > 0) {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'activo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    } else {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'inactivo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    }

                                                    $array_compra_proidmes[] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                        'cantidad' => ($r > 0) ? $r : 0
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => $r,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                }
                                            }
                                        } else {
                                            if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) >= 0)) {
                                                $key = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp);
                                                $cant = convertir_a_float($array_compra_proidDisp[$key]['cantidad']);

                                                if ($cant > 0) {
                                                    $key2 = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo);
                                                    $act = $array_compras_proidNo[$key2]['activo'];
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if (($r > 0) && ($act === 'activo')) {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'activo';
                                                    } else {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'inactivo';
                                                    }
                                                } else {
                                                    $array_compra_proidmes = [];
                                                }
                                            }
                                        }
                                    }

                                    if ((!in_array($row['mes'] . $row['producto'], $array_notas_proid))) {


                                        $array_notas[$row['mes']]['productos'][$row['producto']] =
                                            [
                                                'cantidad'      => $row['cantidad'],
                                                'div_locale'    => $divisa['div_locale'],
                                                'div_simbolo'   => $divisa['div_simbolo'],
                                                'mes'           => $row['mes'],
                                                'costo'         => $row['total_costo'],
                                                'venta'         => $row['total_venta'],
                                                'ganancia'      => $row['total_venta'] - $row['total_costo'],
                                                'porcentaje'    => $porc,
                                                'producto'      => $pro['pro_descripcion'] . '-' . $pro['pro_nombrecorto'],
                                                'id'            => $pro['pro_id'],
                                                'precio'        => $precio,
                                                'codigo'        => $pro['pro_codigo'],
                                                'sucursal'      => $row['su_nombre'],
                                                'facturas'      => $array_compra_proidmes
                                            ];

                                        $array_notas_proid[] = $row['mes'] . $row['producto'];
                                    } else {


                                        if ((in_array($row['mes'] . $row['producto'], $array_notas_proid))) {
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['cantidad'] += $row['cantidad'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['costo']      += $row['total_costo'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['venta']      += $row['total_venta'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['ganancia']   += $row['total_venta'] - $row['total_costo'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['porcentaje'] += $porc;
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['facturas'] = $array_compra_proidmes;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                if ($costo > 0 && $venta > 0) {
                    $ganancia = $venta - $costo;
                    $porcentaje = round((($ganancia) * 100 / $venta), 2);
                }

                Http::json_response([
                    'costo'         => Moneda::moneda($costo, $divisa['div_locale'], $divisa['div_simbolo']),
                    'venta'         => Moneda::moneda($venta, $divisa['div_locale'], $divisa['div_simbolo']),
                    'ganancia'      => Moneda::moneda($ganancia, $divisa['div_locale'], $divisa['div_simbolo']),
                    'porcentaje'    => round($porcentaje),
                    'art_total'     => $artic_total,
                    'meses'         => $array_notas,
                    'ddd'           => $array_compra_proidDisp
                ]);
            }

            Http::json_response([
                'errors'   => $this->array_errors,
            ]);
        }
    }

    public function productosVendidosComprasMesm()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $anio   = filter_input(INPUT_GET, 'anio', FILTER_SANITIZE_NUMBER_INT);
            $mes   = filter_input(INPUT_GET, 'mes', FILTER_SANITIZE_NUMBER_INT);
            $div_id = filter_input(INPUT_GET, 'divisa', FILTER_SANITIZE_NUMBER_INT);
            $sucursal = filter_input(INPUT_GET, 'sucursal', FILTER_SANITIZE_NUMBER_INT);

            if (empty($anio))
                $anio = date('Y');

            if (empty($this->array_errors)) {

                $array_notas        = [];
                $array_notas_id     = [];
                $artic_total        = 0;
                $costo              = 0;
                $venta              = 0;
                $ganancia           = 0;
                $porcentaje         = 0;
                $array_notas_proid  = [];
                $array_compras_proidNo = [];
                $array_compra_proidDisp  = [];

                $productos =  $this->ingresosegresos_model->obtenerProductossinServicios();

                if (is_countable($productos) && count($productos)) {

                    foreach ($productos as $pro) {
                        $array_compra_proidmes = [];

                        $facturas = $this->ingresosegresos_model->obtenerProduccionFACTURASVMesCompras($anio, "01", $div_id, $sucursal, $pro['pro_id']);
                        $divisa = $this->ingresosegresos_model->obtenerDivisa($div_id);
                        $compras = $this->ingresosegresos_model->obtenerFacturasComprasTodas($pro['pro_id']);

                        if (is_countable($facturas) && count($facturas)) {

                            foreach ($facturas as $row) {

                                if ((!in_array($row['mes'], $array_notas_id))) {

                                    if ($row['total_venta'] > 0) {

                                        if ($this->div_master['div_id'] != $div_id) {
                                            $div_factor = $this->ingresosegresos_model->divisaFactor($this->div_master['div_id'], $div_id);
                                            $row['total_costo'] *= $div_factor['df_factor'];
                                        }

                                        $porc = round((($row['total_venta'] - $row['total_costo']) * 100 / $row['total_venta']), 2);
                                    } else {
                                        $porc = 0;
                                    }

                                    foreach ($compras as $row1) {

                                        if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) === 'vacio') && ($this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo) === 'vacio')) {
                                            $cant = convertir_a_float($row1['cmpart_procantidad']);

                                            if ($cant > 0) {
                                                if ($cant == convertir_a_float($row['cantidad'])) {
                                                    $array_compras_proidNo[] = [
                                                        'id'       => $row1['codigo'] . $row['producto'],
                                                        'activo'   => 'inactivo',
                                                        'cantidad' => $cant,
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];
                                                    $array_compra_proidmes[$row['mes'] . $row['producto']] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => 0,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                } else {
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if ($r > 0) {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'activo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    } else {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'inactivo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    }

                                                    $array_compra_proidmes[$row['mes'] . $row['producto']] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => $r,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                }
                                            }
                                        } else {
                                            if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) >= 0)) {
                                                $key = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp);
                                                $cant = convertir_a_float($array_compra_proidDisp[$key]['cantidad']);

                                                if ($cant > 0) {
                                                    $key2 = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo);
                                                    $act = $array_compras_proidNo[$key2]['activo'];
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if (($r > 0) && ($act === 'activo')) {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'activo';
                                                    } else {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'inactivo';
                                                    }
                                                } else {
                                                    $array_compra_proidmes = [];
                                                }
                                            }
                                        }
                                    }

                                    $dta = [];
                                    $dta[$row['producto']] = [
                                        'cantidad'      => $row['cantidad'],
                                        'div_locale'    => $divisa['div_locale'],
                                        'div_simbolo'   => $divisa['div_simbolo'],
                                        'mes'           => $row['mes'],
                                        'costo'         => $row['total_costo'],
                                        'venta'         => $row['total_venta'],
                                        'ganancia'      => $row['total_venta'] - $row['total_costo'],
                                        'porcentaje'    => $porc,
                                        'id'            => $pro['pro_id'],
                                        'producto'      => $pro['pro_descripcion'] . '-' . $pro['pro_nombrecorto'],
                                        'sucursal'      => $row['su_nombre'],
                                        'facturas'      => $array_compra_proidmes
                                    ];

                                    // $data[$row['producto']] = $dta;

                                    //if ((!in_array($row['mes'] . $row['producto'], $array_notas_proid))) {
                                    $array_notas[$row['mes']] = [
                                        'productos'     => $dta,
                                        'mes'           => $row['mes'],
                                    ];

                                    $array_notas_id[] = $row['mes'];
                                    $array_notas_proid[] = $row['mes'] . $row['producto'];
                                    //}
                                } else {

                                    if ($row['total_venta'] > 0) {

                                        if ($this->div_master['div_id'] != $div_id) {
                                            $div_factor = $this->ingresosegresos_model->divisaFactor($this->div_master['div_id'], $div_id);
                                            $row['total_costo'] *= $div_factor['df_factor'];
                                        }

                                        $porc = round((($row['total_venta'] - $row['total_costo']) * 100 / $row['total_venta']), 2);
                                    } else {
                                        $porc = 0;
                                    }

                                    foreach ($compras as $row1) {

                                        if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) === 'vacio') && ($this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo) === 'vacio')) {
                                            $cant = convertir_a_float($row1['cmpart_procantidad']);

                                            if ($cant > 0) {
                                                if ($cant == convertir_a_float($row['cantidad'])) {
                                                    $array_compras_proidNo[] = [
                                                        'id'       => $row1['codigo'] . $row['producto'],
                                                        'activo'   => 'inactivo',
                                                        'cantidad' => $cant,
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];
                                                    $array_compra_proidmes[$row['mes'] . $row['producto']] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => 0,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                } else {
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if ($r > 0) {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'activo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    } else {
                                                        $array_compras_proidNo[] = [
                                                            'id' => $row1['codigo'] . $row['producto'],
                                                            'activo' => 'inactivo',
                                                            'cantidad' => $r,
                                                            'correlativo' => $row1['cmp_correlativo'],
                                                            'codigo' => $row1['codigo'],
                                                        ];
                                                    }

                                                    $array_compra_proidmes[$row['mes'] . $row['producto']] = [
                                                        'correlativo' => $row1['cmp_correlativo'],
                                                        'uri'         => ruta_base() . 'cmpfacturaeditar/verFactura/' .  $row1['codigo'],
                                                        'codigo' => $row1['codigo'],
                                                    ];

                                                    $array_compra_proidDisp[] = [
                                                        'cantidad' => $r,
                                                        'id' => $row1['codigo'] . $row['producto'],
                                                    ];
                                                }
                                            }
                                        } else {
                                            if (($this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp) >= 0)) {
                                                $key = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compra_proidDisp);
                                                $cant = convertir_a_float($array_compra_proidDisp[$key]['cantidad']);

                                                if ($cant > 0) {
                                                    $key2 = $this->searc(strval($row1['codigo'] . $row['producto']), $array_compras_proidNo);
                                                    $act = $array_compras_proidNo[$key2]['activo'];
                                                    $r = ($cant - convertir_a_float($row['cantidad']) > 0) ? ($cant - convertir_a_float($row['cantidad'])) : 0;
                                                    if (($r > 0) && ($act === 'activo')) {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'activo';
                                                    } else {
                                                        $array_compra_proidDisp[$key]['cantidad'] = $r;
                                                        $array_compras_proidNo[$key2]['activo'] = 'inactivo';
                                                    }
                                                } else {
                                                    $array_compra_proidmes = [];
                                                }
                                            }
                                        }
                                    }

                                    if ((!in_array($row['mes'] . $row['producto'], $array_notas_proid))) {


                                        $array_notas[$row['mes']]['productos'][$row['producto']] =
                                            [
                                                'cantidad'      => $row['cantidad'],
                                                'div_locale'    => $divisa['div_locale'],
                                                'div_simbolo'   => $divisa['div_simbolo'],
                                                'mes'           => $row['mes'],
                                                'costo'         => $row['total_costo'],
                                                'venta'         => $row['total_venta'],
                                                'ganancia'      => $row['total_venta'] - $row['total_costo'],
                                                'porcentaje'    => $porc,
                                                'producto'      => $pro['pro_descripcion'] . '-' . $pro['pro_nombrecorto'],
                                                'id'            => $pro['pro_id'],
                                                'sucursal'      => $row['su_nombre'],
                                                'facturas'      => $array_compra_proidmes
                                            ];

                                        $array_notas_proid[] = $row['mes'] . $row['producto'];
                                    } else {


                                        if ((in_array($row['mes'] . $row['producto'], $array_notas_proid))) {
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['cantidad'] += $row['cantidad'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['costo']      += $row['total_costo'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['venta']      += $row['total_venta'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['ganancia']   += $row['total_venta'] - $row['total_costo'];
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['porcentaje'] += $porc;
                                            $array_notas[$row['mes']]['productos'][$row['producto']]['facturas'] = $array_compra_proidmes;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                if ($costo > 0 && $venta > 0) {
                    $ganancia = $venta - $costo;
                    $porcentaje = round((($ganancia) * 100 / $venta), 2);
                }

                Http::json_response([
                    'costo'         => Moneda::moneda($costo, $divisa['div_locale'], $divisa['div_simbolo']),
                    'venta'         => Moneda::moneda($venta, $divisa['div_locale'], $divisa['div_simbolo']),
                    'ganancia'      => Moneda::moneda($ganancia, $divisa['div_locale'], $divisa['div_simbolo']),
                    'porcentaje'    => round($porcentaje),
                    'art_total'     => $artic_total,
                    'meses'         => $array_notas,
                    'ddd'           => $array_compra_proidDisp
                ]);
            }

            Http::json_response([
                'errors'   => $this->array_errors,
            ]);
        }
    }

    private function obtenerCMPPP()
    {
        $ventas_pendientes      = 0;
        $facturas_divisas       = [];
        $facturas_divisas_id    = [];
        $notas_divisas          = [];
        $notas_divisas_id       = [];
        $array_notas            = [];
        $array_factura          = [];
        $totales_divisas = [];
        $total_divisa_id = [];

        $notas = $this->ingresosegresos_model->obtenerNotasPendientesCMP();
        $facturas = $this->ingresosegresos_model->obtenerFacturasPendientesCMP();

        if (is_countable($notas) && count($notas)) {

            foreach ($notas as $row_nota) {

                $acumulado = $this->ingresosegresos_model->obtenerAcumulado($row_nota['cmpn_id'], 9);
                $acumulado['total'] = $acumulado['total'] ?? 0;

                $array_notas[] = [
                    'id'            => $row_nota['cmpn_id'],
                    'correlativo'   => $row_nota['cmpn_correlativo'],
                    'total'         => $row_nota['cmpn_total'],
                    'estatus'       => $row_nota['cmpn_estatus'],
                    'fecha'         => $row_nota['fecha'],
                    'divisa'        => $row_nota['div_nombre'],
                    'div_locale'    => $row_nota['div_locale'],
                    'div_simbolo'   => $row_nota['div_simbolo'],
                    'prov_id'       => $row_nota['prov_id'],
                    'prov_nombre'   => $row_nota['prov_razonsocial'],
                    'acumulado'     => $acumulado['total'] * -1,
                    'pendiente'     => $row_nota['cmpn_total'] + $acumulado['total']
                ];

                $ventas_pendientes += 1;

                if (!in_array($row_nota['div_id'], $notas_divisas_id)) {
                    $notas_divisas[$row_nota['div_id']] = [
                        'div_id'        => $row_nota['div_id'],
                        'div_nombre'        => $row_nota['div_nombre'],
                        'div_locale'    => $row_nota['div_locale'],
                        'div_simbolo'  => $row_nota['div_simbolo'],
                        'total'         => $row_nota['cmpn_total'] + $acumulado['total']
                    ];
                    $notas_divisas_id[] = $row_nota['div_id'];
                } else {
                    $notas_divisas[$row_nota['div_id']]['total'] += $row_nota['cmpn_total'] + $acumulado['total'];
                }
            }
        }

        if (is_countable($facturas) && count($facturas)) {

            foreach ($facturas as $row_factura) {

                $acumulado = $this->ingresosegresos_model->obtenerAcumulado($row_factura['cmp_id'], 8);
                $acumulado['total'] = $acumulado['total'] ?? 0;

                $array_factura[] = [
                    'id'            => $row_factura['cmp_id'],
                    'correlativo'   => $row_factura['cmp_correlativo'],
                    'total'         => $row_factura['cmp_total'],
                    'estatus'       => $row_factura['cmp_estatus'],
                    'fecha'         => $row_factura['fecha'],
                    'divisa'        => $row_factura['div_nombre'],
                    'div_locale'    => $row_factura['div_locale'],
                    'div_simbolo'   => $row_factura['div_simbolo'],
                    'prov_id'       => $row_factura['prov_id'],
                    'prov_nombre'   => $row_factura['prov_razonsocial'],
                    'acumulado'     => $acumulado['total'] * -1,
                    'pendiente'     => $row_factura['cmp_total'] + $acumulado['total']
                ];

                $ventas_pendientes += 1;

                if (!in_array($row_factura['div_id'], $facturas_divisas_id)) {
                    $facturas_divisas[$row_factura['div_id']] = [
                        'div_id'        => $row_factura['div_id'],
                        'div_locale'    => $row_factura['div_locale'],
                        'div_nombre'        => $row_factura['div_nombre'],
                        'div_simbolo'  => $row_factura['div_simbolo'],
                        'total'         => $row_factura['cmp_total'] + $acumulado['total']
                    ];
                    $facturas_divisas_id[] = $row_factura['div_id'];
                } else {
                    $facturas_divisas[$row_factura['div_id']]['total'] += $row_factura['cmp_total'] + $acumulado['total'];
                }
            }
        }

        if (is_countable($facturas_divisas) && count($facturas_divisas)) {
            foreach ($facturas_divisas as $row_fac_div) {
                if (!in_array($row_fac_div['div_id'], $total_divisa_id)) {
                    $totales_divisas[$row_fac_div['div_id']] = [
                        'div_id'        => $row_fac_div['div_id'],
                        'div_locale'    => $row_fac_div['div_locale'],
                        'div_simbolo'  => $row_fac_div['div_simbolo'],
                        'total'         => $row_fac_div['total'],
                    ];
                    $total_divisa_id[] = $row_fac_div['div_id'];
                } else {
                    $totales_divisas[$row_fac_div['div_id']]['total'] += $row_fac_div['total'];
                }
            }
        }

        if (is_countable($notas_divisas) && count($notas_divisas)) {
            foreach ($notas_divisas as $row_n_div) {
                if (!in_array($row_n_div['div_id'], $total_divisa_id)) {
                    $totales_divisas[$row_n_div['div_id']] = [
                        'div_id'        => $row_n_div['div_id'],
                        'div_locale'    => $row_n_div['div_locale'],
                        'div_simbolo'  => $row_n_div['div_simbolo'],
                        'total'         => $row_n_div['total'],
                    ];
                    $total_divisa_id[] = $row_n_div['div_id'];
                } else {
                    $totales_divisas[$row_n_div['div_id']]['total'] += $row_n_div['total'];
                }
            }
        }

        return [
            'facturas_pendiente'    => $array_factura,
            'notas_pendiente'       => $array_notas,
            'totales_divisas_fac'   => $facturas_divisas,
            'totales_divisas_not'   => $notas_divisas,
            'divisas_totales'       => $totales_divisas,
        ];
    }

    public function proveedoresReportesFrecuentes()
    {

        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {


            $proveedor = filter_input(INPUT_GET, 'proveedor', FILTER_SANITIZE_NUMBER_INT);
            $fecha_desde   = filter_input(INPUT_GET, 'fecha_desde', FILTER_SANITIZE_NUMBER_INT);
            $fecha_hasta    = filter_input(INPUT_GET, 'fecha_hasta', FILTER_SANITIZE_NUMBER_INT);
            $divisa    = filter_input(INPUT_GET, 'divisa', FILTER_SANITIZE_NUMBER_INT);

            $artic_total        = 0;
            $costo_total        = 0;
            $venta_total        = 0;
            $totales_notas  = [];
            $subtotales_notas  = [];
            $total_notas_id = [];
            $ganancia = 0;

            $notas =  $this->compras->obtenerCierreMovimientocompletoNot($fecha_desde, $fecha_hasta, $proveedor, $divisa);
            $factura =  $this->compras->obtenerCierreMovimientocompletoFact($fecha_desde, $fecha_hasta, $proveedor, $divisa);
            if ($divisa > 0) {
                $divisa_m = $this->compras->obtenerDivisa($divisa);
            } else {
                $divisa_m = $this->compras->obtenerDivisa($this->div_master['div_id']);
            }

            if (is_countable($notas) && count($notas)) {
                foreach ($notas as $row) {

                    if ($row['fab_divisa'] != $this->div_master['div_id'] && $divisa == 0) {
                        $factor = $this->compras->divisaFactor($row['fab_divisa'], $this->div_master['div_id']);
                        $row['fab_monto'] = round(($row['fab_monto'] * $factor['df_factor']), 2);
                    }

                    if ($row['fab_divisa'] != $divisa && $divisa > 0) {
                        $factor = $this->compras->divisaFactor($row['fab_divisa'], $divisa);
                        $row['fab_monto'] = round(($row['fab_monto'] * $factor['df_factor']), 2);
                    }

                    if (!in_array($row['prov_id'], $total_notas_id)) {
                        $totales_notas[$row['prov_id']] = [
                            'ruta_documento'    => ruta_base() . 'cmpnotaseditar/verFactura/' . $row['cmpn_id'],
                            'div_locale'        => $divisa_m['div_locale'],
                            'div_simbolo'       => $divisa_m['div_simbolo'],
                            'div_nombre'        => $divisa_m['div_nombre'],
                            'proveedor_id'      => $row['prov_id'],
                            'correlativo'       => 'NCV' . $row['cmpn_correlativo'],
                            'venta'             => $row['fab_monto'] * (-1),
                            'sucursal'          => $row['su_nombre'],
                            'proveedor'         => $row['prov_razonsocial'],
                        ];
                        $total_notas_id[] = $row['prov_id'];
                    } else {
                        $totales_notas[$row['prov_id']]['venta']          +=  $row['fab_monto'] * (-1);
                    }

                    $subtotales_notas[] = [
                        'ruta_documento'    => ruta_base() . 'cmpnotaseditar/verFactura/' . $row['cmpn_id'],
                        'div_locale'        => $divisa_m['div_locale'],
                        'div_simbolo'       => $divisa_m['div_simbolo'],
                        'div_nombre'        => $divisa_m['div_nombre'],
                        'proveedor_id'      => $row['prov_id'],
                        'correlativo'       => 'NCV' . $row['cmpn_correlativo'],
                        'venta'             => $row['fab_monto'] * (-1),
                        'fecha'             => $row['fab_fecha'],
                        'sucursal'          => $row['su_nombre'],
                        'proveedor'         => $row['prov_razonsocial'],
                    ];
                }
            }
            if (is_countable($factura) && count($factura)) {
                foreach ($factura as $row) {
                    if ($row['fab_divisa'] != $this->div_master['div_id'] && $divisa == 0) {
                        $factor = $this->compras->divisaFactor($row['fab_divisa'], $this->div_master['div_id']);
                        $row['fab_monto'] = round(($row['fab_monto'] * $factor['df_factor']), 2);
                    }

                    if ($row['fab_divisa'] != $divisa && $divisa > 0) {
                        $factor = $this->compras->divisaFactor($row['fab_divisa'], $divisa);
                        $row['fab_monto'] = round(($row['fab_monto'] * $factor['df_factor']), 2);
                    }

                    if (!in_array($row['prov_id'], $total_notas_id)) {
                        $totales_notas[$row['prov_id']] = [
                            'ruta_documento'    => ruta_base() . 'cmpfacturaeditar/verFactura/' . $row['cmp_id'],
                            'div_locale'        => $divisa_m['div_locale'],
                            'div_simbolo'       => $divisa_m['div_simbolo'],
                            'div_nombre'        => $divisa_m['div_nombre'],
                            'proveedor_id'      => $row['prov_id'],
                            'correlativo'       => 'FACC' . $row['cmp_correlativo'],
                            'venta'             => $row['fab_monto'] * (-1),
                            'sucursal'          => $row['su_nombre'],
                            'proveedor'         => $row['prov_razonsocial'],
                        ];
                        $total_notas_id[] = $row['prov_id'];
                    } else {
                        $totales_notas[$row['prov_id']]['venta']          +=  $row['fab_monto'] * (-1);
                    }

                    $subtotales_notas[] = [
                        'ruta_documento'    => ruta_base() . 'cmpfacturaeditar/verFactura/' . $row['cmp_id'],
                        'div_locale'        => $divisa_m['div_locale'],
                        'div_simbolo'       => $divisa_m['div_simbolo'],
                        'div_nombre'        => $divisa_m['div_nombre'],
                        'proveedor_id'      => $row['prov_id'],
                        'correlativo'       => 'FACC' . $row['cmp_correlativo'],
                        'venta'             => $row['fab_monto'] * (-1),
                        'fecha'             => $row['fab_fecha'],
                        'sucursal'          => $row['su_nombre'],
                        'proveedor'         => $row['prov_razonsocial'],
                    ];
                }
            }


            Http::json_response([
                'proveedores'     => $subtotales_notas,
                'totales' => $totales_notas,
                'ganancia' => $artic_total,
                'costo' => $costo_total,
                'venta' => $venta_total,
                'div_locale'        => $divisa_m['div_locale'],
                'div_simbolo'       => $divisa_m['div_simbolo'],
                'div_nombre'        => $divisa_m['div_nombre'],
            ]);
        }
    }

    public function reporteGeneralAnual()
    {

        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {


            $anio   = filter_input(INPUT_POST, 'anio', FILTER_SANITIZE_NUMBER_INT);

            $array_notas        = [];
            $array_notas_id     = [];
            $array_documentos_id     = [];
            $artic_total        = 0;
            $costo              = 0;
            $venta              = 0;

            if ($anio == 0) {
                $facturas = $this->compras->obtenerFacturasAnualesTodos();
                $notas = $this->compras->obtenerNotasdeEntregaAnualesTodos();
            } else {
                $facturas = $this->compras->obtenerFacturasAnualest("YEAR(cmp.cmp_fecha)=$anio");
                $notas = $this->compras->obtenerNotasdeEntregaAnualest("YEAR(cmp.cmpn_fecha)=$anio");
            }


            if (is_countable($notas) && count($notas)) {

                foreach ($notas as $row) {

                    if ($row['divisa'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['divisa'], $this->div_master['div_id']);
                        $row['total'] = round(($row['total'] * $factor['df_factor']), 2);
                    }

                    if (!in_array($row['anio'], $array_notas_id)) {

                        $array_notas[$row['anio']] = [
                            'anio'  => $row['anio'],
                            'venta'         => $row['total'],
                            'documentos'    => 1
                        ];
                        $array_notas_id[] = $row['anio'];
                    } else {
                        $array_notas[$row['anio']]['venta']      += $row['total'];
                        $array_notas[$row['anio']]['documentos']      += 1;
                    }

                    $artic_total    += 1;
                    $venta          += $row['total'];
                }
            }

            if (is_countable($facturas) && count($facturas)) {

                foreach ($facturas as $row) {

                    if ($row['divisa'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['divisa'], $this->div_master['div_id']);
                        $row['total'] = round(($row['total'] * $factor['df_factor']), 2);
                    }

                    if (!in_array($row['anio'], $array_notas_id)) {
                        $array_notas[$row['anio']] = [
                            'anio'  => $row['anio'],
                            'venta'         => $row['total'],
                            'documentos'    => 1
                        ];
                        $array_notas_id[] = $row['anio'];
                    } else {
                        $array_notas[$row['anio']]['venta']      += $row['total'];
                        $array_notas[$row['anio']]['documentos']      += 1;
                    }

                    $artic_total    += 1;
                    $venta          += $row['total'];
                }
            }

            Http::json_response([
                'anios'   => $array_notas,
                'total' => $venta,
                'div_locale'    => $this->div_master['div_locale'],
                'div_simbolo'  => $this->div_master['div_simbolo'],
            ]);
        }
    }

    public function reporteGeneralMensual()
    {

        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {


            $anio   = filter_input(INPUT_POST, 'anio', FILTER_SANITIZE_NUMBER_INT);

            $array_notas        = [];
            $array_notas_id     = [];
            $array_documentos_id     = [];
            $artic_total        = 0;
            $costo              = 0;
            $venta              = 0;

            // if($anio>0){
            //    $facturas = $this->compras->obtenerFacturasAnualest("YEAR(cmp.cmp_fecha)=$anio");
            //    $notas = $this->compras->obtenerNotasdeEntregaAnualest("YEAR(cmp.cmpn_fecha)=$anio");
            // }else{
            $facturas = $this->compras->obtenerFacturasAnualesMes("YEAR(cmp.cmp_fecha)=$anio");
            $notas = $this->compras->obtenerNotasdeEntregaAnualesMes("YEAR(cmp.cmpn_fecha)=$anio");
            //}


            if (is_countable($notas) && count($notas)) {

                foreach ($notas as $row) {

                    if ($row['divisa'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['divisa'], $this->div_master['div_id']);
                        $row['total'] = round(($row['total'] * $factor['df_factor']), 2);
                    }

                    if (!in_array($row['mes'], $array_notas_id)) {

                        $array_notas[$row['mes']] = [
                            'mes'  => $row['mes'],
                            'venta'         =>  round($row['total'], 2),
                            'documentos'    => 1
                        ];
                        $array_notas_id[] = $row['mes'];
                    } else {
                        $array_notas[$row['mes']]['venta']      +=  round($row['total'], 2);
                        $array_notas[$row['mes']]['documentos']      += 1;
                    }

                    $artic_total    += 1;
                    $venta          += round($row['total'], 2);
                }
            }

            if (is_countable($facturas) && count($facturas)) {

                foreach ($facturas as $row) {

                    if ($row['divisa'] != $this->div_master['div_id']) {
                        $factor = $this->compras->divisaFactor($row['divisa'], $this->div_master['div_id']);
                        $row['total'] = round(($row['total'] * $factor['df_factor']), 2);
                    }

                    if (!in_array($row['mes'], $array_notas_id)) {
                        $array_notas[$row['mes']] = [
                            'mes'  => $row['mes'],
                            'venta'         =>  round($row['total'], 2),
                            'documentos'    => 1
                        ];
                        $array_notas_id[] = $row['mes'];
                    } else {
                        $array_notas[$row['mes']]['venta']      += round($row['total'], 2);
                        $array_notas[$row['mes']]['documentos']      += 1;
                    }

                    $artic_total    += 1;
                    $venta          +=  round($row['total'], 2);
                }
            }

            Http::json_response([
                'meses'   => $array_notas,
                'total' => $venta,
                'div_locale'    => $this->div_master['div_locale'],
                'div_simbolo'  => $this->div_master['div_simbolo'],
            ]);
        }
    }

    public function obtenerDatosCrudForaneos()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $newProv = new Proveedor();

            Http::json_response([
                'proveedores' => $newProv->proveedoresActivos(),
                'series'        => $this->compras->Series()
            ]);
        }
    }

    public function obtenerProductosStockBajoProv()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $lista = [];
            $lista_productos = [];

            $prov   = filter_input(INPUT_POST, 'prov', FILTER_SANITIZE_NUMBER_INT);
            $almacen   = filter_input(INPUT_POST, 'almacen', FILTER_SANITIZE_NUMBER_INT);

            $productos = new Productos();
            $nombre = '';

            $lista = $productos->obtenerProductosStockBajoComp($prov, $almacen);

            if (is_countable($lista) && count($lista)) {
                foreach ($lista as $row) {
                    $lista_productos[] = [
                        'id' => $row['pro_id'],
                        'prov' => $row['pp_provid'],
                        'descripcion' => $row['pro_descripcion'],
                        'cantidad_disponible' => $row['ps_cantidad'],
                        'cantidad' => 1,
                        'min' => $row['ps_stockmin'],
                        'max' => $row['ps_stockmax'],
                        'checked' => false,
                    ];
                    $nombre = $row['prov_nombre'];
                }
            }

            Http::json_response([
                'stock' => $lista_productos,
                'nombre' => $nombre
            ]);
        }
    }

    public function CrearPedido()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id_cliente = $this->solicitud->sanitize($_POST['prov'], FILTER_SANITIZE_NUMBER_INT);
            $almacen = $this->solicitud->sanitize($_POST['almacen'], FILTER_SANITIZE_NUMBER_INT);
            $id_serie = $this->solicitud->sanitize($_POST['id_serie'], FILTER_SANITIZE_NUMBER_INT);

            $carritoCompra = new CarritoCompras();
            $carritoModel = new CarritoVentasModel();

            $proveedor          = $carritoModel->obtenerProveedor($id_cliente);
            $serie              = $carritoModel->obtenerSerie($id_serie);

            $tipo_doc = 10;
            $tabla_db_articulos = 'app_cmppedidos_articulos';
            $uri = 'cmppedidoseditar/verFactura/';
            $correlativo = $carritoModel->chuequearCorrelativo($id_serie, $tipo_doc);

            if (!$correlativo) {
                $this->array_errors[] = 'No existe correlativo entre la serie seleccionada y el documento Factura por favor dirijase a Modulo Configuración -> correlativos';
            }

            if (empty($this->array_errors)) {

                $fecha = date('Y-m-d');
                $contenido_carrito =  json_decode($_POST['productos'], true);
                $fecha = date('Y-m-d', strtotime($fecha)) . ' ' . date('H:i:s');
                $factor = 1;

                $subtotal = 0;
                $neto = 0;
                $total = 0;
                $cantidad = 0;

                $newProductos = new Productos();

                foreach ($contenido_carrito as $row) {

                    if ($row['checked'] == true) {
                        $producto = $newProductos->obtenerProductoPorId($row['id']);

                        $subtotal += $producto['pro_costo'] * $row['cantidad'];
                        $neto += $producto['pro_costo'] * $row['cantidad'];
                        $cantidad += $row['cantidad'];
                    }
                }

                $data_form_master = [
                    'cmppe_subtotal'           => $subtotal,
                    'cmppe_descuento'          => 0,
                    'cmppe_neto'               => $neto,
                    'cmppe_iva'                => 0,
                    'cmppe_total'              => $subtotal,
                    'cmppe_articulos_total'    => $cantidad,
                    'cmppe_empresaid'          => 1,
                    'cmppe_empleadoid'         => $_SESSION['user_data']['einfo_id'],
                    'cmppe_divid'              => 2,
                    'cmppe_serid'              => $serie['ser_id'],
                    'cmppe_serie'              => $serie['ser_descripcion'],
                    'cmppe_fecha'              => $fecha,
                    'cmppe_estatus'            => 2,
                    'cmppe_correlativo'        => $correlativo['cor_correlativo'] + 1,
                    'cmppe_cliid'              => $proveedor['prov_id'],
                    'cmppe_env_nombres'       => $proveedor['prov_razonsocial'],
                    'cmppe_env_direccion'     => $proveedor['prov_direccion'],
                    'cmppe_env_pais'          => $proveedor['prov_paisid'],
                    'cmppe_env_estado'        => $proveedor['prov_estado'],
                    'cmppe_env_ciudad'        => $proveedor['prov_ciudad'],
                    'cmppe_env_codigo_postal' => $proveedor['prov_codigo_postal'],
                    'cmppe_observaciones'      => '',
                    'cmppe_almid'              => $almacen,
                    'cmppe_descontado'         => 0,
                ];
                $data_form_articulos = [
                    'cmppeart_peid', 'cmppeart_proid', 'cmppeart_procodigo', 'cmppeart_propnombre',
                    'cmppeart_proprecio', 'cmppeart_procantidad', 'cmppeart_prodescuento',
                    'cmppeart_proneto', 'cmppeart_idiva', 'cmppeart_almid', 'cmppeart_proiva', 'cmppeart_ivatotal',
                    'cmppeart_prototal', 'cmppeart_descontar_inventario'
                ];

                try {

                    $carritoModel->transactionBegin();

                    $id_venta = $carritoModel->guardar($data_form_master, false, "app_cmppedidos");

                    $carritoModel->actualizarCorrelativo($serie['ser_id'], $tipo_doc);

                    foreach ($contenido_carrito as $row) {

                        if ($row['checked'] == true) {
                            $almacens    = $carritoModel->obtenerAlmacen($almacen);
                            //$cantidad   = $carritoModel->chuequearStock($row['id'], $almacen);

                            $producto = $newProductos->obtenerProductoPorId($row['id']);
                            $pron = substr($producto['pro_descripcion'], 0, 100);

                            $carritoModel->guardar([
                                $data_form_articulos[0] => $id_venta,
                                $data_form_articulos[1] => $row['id'],
                                $data_form_articulos[2] => $producto['pro_codigo'],
                                $data_form_articulos[3] => $pron,
                                $data_form_articulos[4] => $producto['pro_costo'],
                                $data_form_articulos[5] => convertir_a_float($row['cantidad']),
                                $data_form_articulos[6] => 0,
                                $data_form_articulos[7] => $producto['pro_costo'] * $row['cantidad'],
                                $data_form_articulos[8] => 2,
                                $data_form_articulos[9] =>  $almacen,
                                $data_form_articulos[10] => 0,
                                $data_form_articulos[11] => 0,
                                $data_form_articulos[12] => $producto['pro_costo'] * $row['cantidad'],
                                $data_form_articulos[13] => 0,
                            ], true, "app_cmppedidos_articulos");
                        }
                    }

                    $carritoModel->transactionCommit();
                } catch (\Throwable $th) {
                    $carritoModel->transactionRollBack();
                    $this->array_errors[] = 'Error en transacción' . $th;
                }

                if (empty($this->array_errors)) {

                    Http::json_response(ruta_base() . $uri . $id_venta);
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function obtenerProductosStockBajo()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $stockBajo = $this->compras->obtenerProductosStockBajo();

            $producto = [];

            if (is_countable($stockBajo) && count($stockBajo) > 0) {
                foreach ($stockBajo as $row) {
                    $f_cantidad = $this->compras->obtenerPreciosFacturasComprasProvCont($row['pro_id']);
                    $n_cantidad = $this->compras->obtenerPreciosNotasComprasProvCont($row['pro_id']);
                    $cant_existente = $this->compras->obtenerProductosStockBajoCantidad($row['pro_id']);

                    $f_t = $this->compras->obtenerPreciosFacturasComprasProvTotal($row['pro_id']);
                    $n_t = $this->compras->obtenerPreciosNotasComprasProvTotal($row['pro_id']);

                    $existe = 0;
                    foreach ($cant_existente as $row1) {
                        $existe += $row1['ps_cantidad'] != null ? $row1['ps_cantidad'] : 0;
                    }

                    $cantidad = 0;
                    $ganancia = 0;
                    foreach ($n_cantidad as $row1) {
                        $cantidad += $row1['cantidad'] != null ? $row1['cantidad'] : 0;
                    }

                    foreach ($f_cantidad as $row1) {
                        $cantidad += $row1['cantidad'] != null ? $row1['cantidad'] : 0;
                    }

                    foreach ($n_t as $row1) {
                        $factor2 = 1;

                        if ($row1['ne_divid'] != $this->div_master['div_id']) {
                            $factor = $this->compras->divisaFactor($row1['ne_divid'], $this->div_master['div_id']);
                            $row1['neart_procosto'] = round(($row1['neart_procosto'] * $factor['df_factor']), 2);
                            $row1['neart_prototal'] = round(($row1['neart_prototal'] * $factor['df_factor']), 2);
                            $factor2 = $factor['df_factor'];
                        }

                        if ($row1['neart_prototal'] > 1) {
                            $ganancia += round($row1['neart_prototal'] - ($row1['neart_procosto'] * $row1['neart_procantidad']), 2);
                            $ga = ($ganancia * 100);
                            $ganc = ($row1['neart_procosto'] * $row1['neart_procantidad']);
                            if ($ga > 0 && $ganc > 0) {
                                $ganancia_porct =  $ga / $ganc;
                            } else {
                                $ganancia_porct = 0;
                            }
                        }
                    }

                    foreach ($f_t as $row1) {
                        $factor2 = 1;

                        if ($row1['fm_divid'] != $this->div_master['div_id']) {
                            $factor = $this->compras->divisaFactor($row1['fm_divid'], $this->div_master['div_id']);
                            $row1['fa_procosto'] = round(($row1['fa_procosto'] * $factor['df_factor']), 2);
                            $row1['fa_prototal'] = round(($row1['fa_prototal'] * $factor['df_factor']), 2);
                            $factor2 = $factor['df_factor'];

                            if ($row1['fa_prototal'] > 1) {
                                $ganancia += round($row1['fa_prototal'] - ($row1['fa_procosto'] * $row1['fa_procantidad']), 2);
                                $ga = ($ganancia * 100);
                                $ganc = ($row1['fa_procosto'] * $row1['fa_procantidad']);
                                if ($ga > 0 && $ganc > 0) {
                                    $ganancia_porct =  $ga / $ganc;
                                } else {
                                    $ganancia_porct = 0;
                                }
                            }
                        }
                    }

                    $producto[] = [
                        "pro_id" => $row['pro_id'],
                        "pro_codigo" => $row['pro_codigo'],
                        "pro_descripcion" => $row['pro_descripcion'],
                        "ps_cantidad" => $existe,
                        "ps_stockmin" => $row['ps_stockmin'],
                        "cantidad" => $cantidad,
                        "ganancias" => $ganancia,
                        "div_locale"        => $this->div_master['div_locale'],
                        "div_simbolo"      => $this->div_master['div_simbolo'],
                        "div_nombre"       => $this->div_master['div_nombre'],
                    ];
                }
            }

            Http::json_response($producto);
        }
    }

    public function obtenerProductosProveedores($id)
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $producto = $this->compras->obtenerProducto($id);

            $notas_s =  $this->compras->obtenerPreciosNotasComprasProv($id);
            $facturas_s = $this->compras->obtenerPreciosFacturasComprasProv($id);
            $proveedor = [];
            $bajo = [];
            $lista_proveedores_sugeridos = [];

            $n = 0;
            $f = 0;

            if (is_countable($notas_s) && count($notas_s) > 0) {
                foreach ($notas_s as $row) {

                    if (!in_array($row['prov_id'], $proveedor)) {
                        $precio = $producto['pro_costo'];
                        $notas = $this->compras->obtenerPrecioNotasCompras($producto['pro_id'], $row['prov_id']);
                        $facturas = $this->compras->obtenerPrecioFacturasCompras($producto['pro_id'], $row['prov_id']);

                        $fact = new DateTime($notas['fecha']);
                        $not = new DateTime($facturas['fecha']);

                        if ($not > $fact) {
                            $precio = $notas['precio'];
                        } else {
                            if ($facturas['divisa'] != null) {
                                if ($facturas['divisa'] != $this->div_master['div_id']) {
                                    $factor = $this->compras->divisaFactor($facturas['divisa'], $this->div_master['div_id']);
                                    $precio = $facturas['precio'];
                                } else {
                                    $precio = $facturas['precio'];
                                }
                            }
                        }

                        $bajo[] = [
                            "pro_id" => $producto['pro_id'],
                            "pro_nombrecorto" => $producto['pro_descripcion'],
                            "pprov_producto" => $producto['pro_descripcion'],
                            "pprov_proveedor" => $row['prov_id'],
                            "prov_razonsocial" => $row['prov_razonsocial'],
                            "pprov_precio" => round($precio, 2),
                            "prov_direccion" => $row['prov_direccion'],
                            "comprador" => 1,
                        ];
                        $proveedor[] = $row['prov_id'];
                    }
                    $n = $n + 1;
                }
            }

            if (is_countable($facturas_s) && count($facturas_s) > 0) {
                foreach ($facturas_s as $row) {

                    if (!in_array($row['prov_id'], $proveedor)) {
                        $precio = $producto['pro_costo'];
                        $notas = $this->compras->obtenerPrecioNotasCompras($producto['pro_id'], $row['prov_id']);
                        $facturas = $this->compras->obtenerPrecioFacturasCompras($producto['pro_id'], $row['prov_id']);

                        $fact = new DateTime($notas['fecha']);
                        $not = new DateTime($facturas['fecha']);

                        if ($not > $fact) {
                            $precio = $notas['precio'];
                        } else {
                            if ($facturas['divisa'] != null) {
                                if ($facturas['divisa'] != $this->div_master['div_id']) {
                                    $factor = $this->compras->divisaFactor($facturas['divisa'], $this->div_master['div_id']);
                                    $precio = $facturas['precio'];
                                } else {
                                    $precio = $facturas['precio'];
                                }
                            }
                        }

                        $bajo[] = [
                            "pro_id" => $producto['pro_id'],
                            "pro_nombrecorto" => $producto['pro_descripcion'],
                            "pprov_producto" => $producto['pro_descripcion'],
                            "pprov_proveedor" => $row['prov_id'],
                            "prov_razonsocial" => $row['prov_razonsocial'],
                            "pprov_precio" => round($precio, 2),
                            "prov_direccion" => $row['prov_direccion'],
                            "comprador" => 1,
                        ];
                        $proveedor[] = $row['prov_id'];
                    }
                    $f = $f + 1;
                }
            }

            $lista_proveedores_sugeridos = $this->compras->obtenerProductosStockBajoId($producto['pro_descripcion']);

            if (count($notas_s) == $n && count($facturas_s) == $f && $proveedor > 0) {
                if (is_countable($lista_proveedores_sugeridos) && count($lista_proveedores_sugeridos) > 0) {
                    foreach ($lista_proveedores_sugeridos as $row) {

                        $palabras = explode(" ", strtolower($row['pprov_producto']));
                        $palabras1 = explode(" ",  strtolower($producto['pro_descripcion']));

                        $r = 0;

                        foreach ($palabras as $i => $pal) {
                            if (in_array($pal, $palabras1)) {
                                $r += 1;
                            }
                        }

                        if (!in_array($row['pprov_proveedor'], $proveedor) && $r >= 2) {
                            $bajo[] = [
                                "pro_id" => $producto['pro_id'],
                                "pro_nombrecorto" => $producto['pro_descripcion'],
                                "pprov_producto" => $row['pprov_producto'],
                                "pprov_proveedor" => $row['pprov_proveedor'],
                                "prov_razonsocial" => $row['prov_razonsocial'],
                                "pprov_precio" => $row['pprov_precio'],
                                "prov_direccion" => $row['prov_direccion'],
                                "comprador" => 0,
                            ];
                        }
                    }
                }
            }

            Http::json_response($bajo);
        }
    }
}
